﻿-- Anti-Patterns and Best Practices

SET SCHEMA DEMOSCHEMA;

-- Anti-Patterns in SQL

-- Anti-pattern: Using SELECT *
-- Problem: Returns unnecessary columns, increases network traffic and resource usage
SELECT * FROM EMPLOYEE WHERE DEPTNO = 'D01';

-- Better: Specify only needed columns
SELECT EMPNO, LASTNAME, FIRSTNAME, JOB_TITLE, SALARY
FROM EMPLOYEE WHERE DEPTNO = 'D01';

-- Anti-pattern: Using string concatenation for date comparisons
-- Problem: Bypasses indexes and causes full table scans
SELECT * FROM ORDERS 
WHERE VARCHAR_FORMAT(ORDERDATE, 'YYYY-MM-DD') = '2023-01-15';

-- Better: Use proper date formatting
SELECT * FROM ORDERS 
WHERE ORDERDATE = DATE('2023-01-15');

-- Anti-pattern: Unnecessary subqueries
-- Problem: Can cause performance issues, especially with large datasets
SELECT o.ORDERNO, o.ORDERDATE, o.CUSTNO
FROM ORDERS o
WHERE o.CUSTNO IN (
  SELECT CUSTNO FROM CUSTOMER WHERE STATE = 'NY'
);

-- Better: Use JOIN
SELECT o.ORDERNO, o.ORDERDATE, o.CUSTNO
FROM ORDERS o
JOIN CUSTOMER c ON o.CUSTNO = c.CUSTNO
WHERE c.STATE = 'NY';


-- Anti-pattern: Not using parameterized queries (pseudocode)
/* In application code:
   String query = "SELECT * FROM CUSTOMER WHERE CUSTNAME = '" + userInput + "'";
   // SQL injection risk!
*/

-- Better: Use parameter markers
/* In application code:
   PreparedStatement ps = conn.prepareStatement(
     "SELECT * FROM CUSTOMER WHERE CUSTNAME = ?"
   );
   ps.setString(1, userInput);
*/

-- Anti-pattern: Scalar functions on indexed columns
-- Problem: Prevents index usage
SELECT * FROM EMPLOYEE 
WHERE UPPER(LASTNAME) = 'SMITH';

-- Better: Create an expression-based index or rewrite the query
CREATE INDEX ix_employee_upper_lastname ON EMPLOYEE (UPPER(LASTNAME));
-- Or use:
SELECT * FROM EMPLOYEE 
WHERE LASTNAME = 'Smith' OR LASTNAME = 'SMITH' OR LASTNAME = 'smith';
 

-- SQL Best Practices
-- Best Practice: Use CTEs for complex queries
WITH order_summary AS (
  SELECT 
    CUSTNO, 
    COUNT(*) AS ORDER_COUNT,
    SUM(TOTAL_AMOUNT) AS TOTAL_SPEND
  FROM ORDERS
  WHERE ORDERDATE >= CURRENT DATE - 1 YEAR
  GROUP BY CUSTNO
),
customer_tier AS (
  SELECT 
    CUSTNO,
    CASE
      WHEN TOTAL_SPEND >= 10000 THEN 'PLATINUM'
      WHEN TOTAL_SPEND >= 5000 THEN 'GOLD'
      WHEN TOTAL_SPEND >= 1000 THEN 'SILVER'
      ELSE 'BRONZE'
    END AS CUSTOMER_TIER
  FROM order_summary
)
SELECT 
  c.CUSTNO,
  c.CUSTNAME,
  os.ORDER_COUNT,
  os.TOTAL_SPEND,
  ct.CUSTOMER_TIER
FROM CUSTOMER c
JOIN order_summary os ON c.CUSTNO = os.CUSTNO
JOIN customer_tier ct ON c.CUSTNO = ct.CUSTNO
ORDER BY os.TOTAL_SPEND DESC;

-- Best Practice: Use MERGE for upsert operations
MERGE INTO CUSTOMER_STATUS cs
USING (
  SELECT 
    c.CUSTNO,
    c.CUSTNAME,
    (SELECT MAX(ORDERDATE) FROM ORDERS WHERE CUSTNO = c.CUSTNO) AS LAST_ORDER_DATE,
    (SELECT SUM(TOTAL_AMOUNT) FROM ORDERS WHERE CUSTNO = c.CUSTNO AND ORDERDATE >= CURRENT DATE - 1 YEAR) AS ANNUAL_SPEND
  FROM CUSTOMER c
) AS source
ON cs.CUSTNO = source.CUSTNO
WHEN MATCHED THEN
  UPDATE SET 
    cs.LAST_ORDER_DATE = source.LAST_ORDER_DATE,
    cs.ANNUAL_SPEND = source.ANNUAL_SPEND,
    cs.STATUS = CASE
      WHEN source.LAST_ORDER_DATE < CURRENT DATE - 6 MONTHS THEN 'INACTIVE'
      WHEN source.ANNUAL_SPEND > 10000 THEN 'PREMIUM'
      ELSE 'ACTIVE'
    END,
    cs.LAST_UPDATED = CURRENT TIMESTAMP
WHEN NOT MATCHED THEN
  INSERT (CUSTNO, CUSTNAME, LAST_ORDER_DATE, ANNUAL_SPEND, STATUS, LAST_UPDATED)
  VALUES (
    source.CUSTNO, 
    source.CUSTNAME, 
    source.LAST_ORDER_DATE,
    source.ANNUAL_SPEND,
    CASE
      WHEN source.LAST_ORDER_DATE < CURRENT DATE - 6 MONTHS THEN 'INACTIVE'
      WHEN source.ANNUAL_SPEND > 10000 THEN 'PREMIUM'
      ELSE 'ACTIVE'
    END,
    CURRENT TIMESTAMP
  );
