﻿-- Replacing RPG with SQL

-- Set-Based Operations
-- RPG approach (pseudocode):
/*
READ CUSTMAST;
DOW NOT %EOF(CUSTMAST);
  IF LAST_ORDER_DATE < %DATE() - %MONTHS(6);
    CUSTSTAT = 'INACTIVE';
    UPDATE CUSTREC;
  ENDIF;
  READ CUSTMAST;
ENDDO;
*/

-- SQL equivalent (set-based operation):
UPDATE CUSTOMER
SET STATUS = 'INACTIVE'
WHERE (
  SELECT MAX(ORDERDATE) 
  FROM ORDERS 
  WHERE CUSTNO = CUSTOMER.CUSTNO
) < CURRENT DATE - 24 MONTHS
OR NOT EXISTS (
  SELECT 1 
  FROM ORDERS 
  WHERE CUSTNO = CUSTOMER.CUSTNO
);
 

-- Complex Business Logic in SQL
-- Update customer discount based on purchasing history
UPDATE CUSTOMER c
SET DISCOUNT_TIER = 
  CASE
    WHEN (
      SELECT SUM(TOTAL_AMOUNT) 
      FROM ORDERS 
      WHERE CUSTNO = c.CUSTNO 
      AND ORDERDATE >= CURRENT DATE - 1 YEAR
    ) >= 25000 THEN 'PLATINUM'
    
    WHEN (
      SELECT SUM(TOTAL_AMOUNT) 
      FROM ORDERS 
      WHERE CUSTNO = c.CUSTNO 
      AND ORDERDATE >= CURRENT DATE - 1 YEAR
    ) >= 10000 THEN 'GOLD'
    
    WHEN (
      SELECT SUM(TOTAL_AMOUNT) 
      FROM ORDERS 
      WHERE CUSTNO = c.CUSTNO 
      AND ORDERDATE >= CURRENT DATE - 1 YEAR
    ) >= 5000 THEN 'SILVER'
    
    WHEN (
      SELECT COUNT(*) 
      FROM ORDERS 
      WHERE CUSTNO = c.CUSTNO 
      AND ORDERDATE >= CURRENT DATE - 1 YEAR
    ) >= 5 THEN 5
    
    ELSE 0
  END
WHERE c.STATUS = 'ACTIVE';
 

-- Stored Procedure for Customer Analysis
CREATE PROCEDURE ANALYZE_ORDER_TRENDS 
(
  P_START_DATE DATE,
  P_END_DATE DATE,
  P_REGION_ID CHAR(2)
)
RESULT SETS 1
BEGIN
  -- Declare cursor for monthly trends
  DECLARE CURSOR1 CURSOR FOR
    SELECT 
      YEAR(O.ORDERDATE) AS ORDER_YEAR,
      MONTH(O.ORDERDATE) AS ORDER_MONTH,
      COUNT(O.ORDERNO) AS ORDER_COUNT,
      SUM(O.TOTAL_AMOUNT) AS TOTAL_SALES,
      COUNT(DISTINCT O.CUSTNO) AS CUSTOMER_COUNT
    FROM ORDERS O
    JOIN CUSTOMER C ON O.CUSTNO = C.CUSTNO
    WHERE (P_START_DATE IS NULL OR O.ORDERDATE >= P_START_DATE)
    AND (P_END_DATE IS NULL OR O.ORDERDATE <= P_END_DATE)
    AND (P_REGION_ID IS NULL OR C.REGION_ID = P_REGION_ID)
    GROUP BY YEAR(O.ORDERDATE), MONTH(O.ORDERDATE)
    ORDER BY YEAR(O.ORDERDATE), MONTH(O.ORDERDATE);
  
  -- Open cursor to return results
  OPEN CURSOR1;
END;

-- Simplified Product Recommendations Function for DB2 for i
CREATE OR REPLACE FUNCTION RECOMMENDED_PRODUCTS (
  P_CUSTNO CHAR(10),
  P_LIMIT INTEGER
)
RETURNS TABLE (
  PRODNO CHAR(10),
  PRODNAME VARCHAR(100),
  CATEGORY VARCHAR(50),
  RECOMMENDATION_TYPE VARCHAR(20),
  SCORE DECIMAL(5,2)
)
LANGUAGE SQL
DETERMINISTIC
NO EXTERNAL ACTION
READS SQL DATA
RETURN
  -- Use a simpler approach with just two recommendation types
  -- 1. Category-based recommendations from customer's purchase history
  SELECT 
    P.PRODNO,
    P.PRODNAME,
    P.CATEGORY,
    'CATEGORY_BASED' AS RECOMMENDATION_TYPE,
    5.0 AS SCORE
  FROM PRODUCT P
  WHERE P.CATEGORY IN (
    -- Find categories this customer has purchased from
    SELECT DISTINCT PR.CATEGORY 
    FROM ORDERS O
    JOIN ORDERLINE OL ON O.ORDERNO = OL.ORDERNO
    JOIN PRODUCT PR ON OL.PRODNO = PR.PRODNO
    WHERE O.CUSTNO = P_CUSTNO
  )
  -- Exclude products the customer already purchased
  AND P.PRODNO NOT IN (
    SELECT OL.PRODNO
    FROM ORDERS O
    JOIN ORDERLINE OL ON O.ORDERNO = OL.ORDERNO
    WHERE O.CUSTNO = P_CUSTNO
  )
  AND P.ACTIVE_FLAG = 'Y'
  
  UNION ALL
  
  -- 2. Popular products recommendation (most-ordered products)
  SELECT 
    P.PRODNO,
    P.PRODNAME,
    P.CATEGORY,
    'POPULAR_PRODUCT' AS RECOMMENDATION_TYPE,
    4.0 AS SCORE
  FROM PRODUCT P
  JOIN (
    SELECT 
      OL.PRODNO,
      SUM(OL.QUANTITY) AS TOTAL_ORDERED
    FROM ORDERLINE OL
    JOIN ORDERS O ON OL.ORDERNO = O.ORDERNO
    GROUP BY OL.PRODNO
    ORDER BY TOTAL_ORDERED DESC
    FETCH FIRST 20 ROWS ONLY
  ) AS POPULAR ON P.PRODNO = POPULAR.PRODNO
  -- Exclude products the customer already purchased
  WHERE P.PRODNO NOT IN (
    SELECT OL.PRODNO
    FROM ORDERS O
    JOIN ORDERLINE OL ON O.ORDERNO = OL.ORDERNO
    WHERE O.CUSTNO = P_CUSTNO
  )
  AND P.ACTIVE_FLAG = 'Y'
  
  -- Sort by score and limit results
  ORDER BY SCORE DESC, PRODNO
  FETCH FIRST P_LIMIT ROWS ONLY;

-- Example usage:
SELECT * FROM TABLE(DEMOSCHEMA.RECOMMENDED_PRODUCTS(P_CUSTNO => 'C000000001', P_LIMIT => 5));
 

-- User-Defined Functions

-- Scalar UDF for Calculating Customer Lifetime Value
CREATE  FUNCTION customer_lifetime_value(
  p_custno CHAR(10),
  p_discount_rate DECIMAL(5,2) DEFAULT 10.00  -- Annual discount rate as a percentage
)
RETURNS DECIMAL(12,2)
LANGUAGE SQL
DETERMINISTIC

RETURN
  WITH yearly_revenue AS (
    SELECT 
      YEAR(ORDERDATE) AS YEAR,
      SUM(TOTAL_AMOUNT) AS ANNUAL_REVENUE
    FROM ORDERS
    WHERE CUSTNO = p_custno
    GROUP BY YEAR(ORDERDATE)
  ),
  lifetime_metrics AS (
    SELECT 
      COUNT(*) AS YEARS_ACTIVE,
      AVG(ANNUAL_REVENUE) AS AVG_ANNUAL_REVENUE,
      (SELECT COUNT(*) FROM ORDERS WHERE CUSTNO = p_custno) AS TOTAL_ORDERS,
      (SELECT MAX(ORDERDATE) FROM ORDERS WHERE CUSTNO = p_custno) AS LAST_ORDER_DATE
    FROM yearly_revenue
  )
  SELECT 
    CASE
      WHEN DAYS(CURRENT DATE) - DAYS(LAST_ORDER_DATE) > 365 THEN 
        -- Customer likely churned, return historical value
        AVG_ANNUAL_REVENUE * YEARS_ACTIVE
      ELSE 
        -- Active customer, project future value
        AVG_ANNUAL_REVENUE * (1 + (100 - p_discount_rate) / 100)
    END
  FROM lifetime_metrics;
 
-- Row and Column Access Control (RCAC)

-- Comprehensive RCAC Implementation
-- Create a permission to restrict data access based on region
CREATE PERMISSION regional_access ON CUSTOMER
FOR ROWS WHERE 
  REGION = (SELECT USER_REGION FROM APP_USERS WHERE USER_ID = SESSION_USER)
  OR VERIFY_GROUP_FOR_USER(SESSION_USER, 'ADMIN') = 1
  OR VERIFY_GROUP_FOR_USER(SESSION_USER, 'GLOBAL_SALES') = 1
ENFORCED FOR ALL ACCESS
ENABLE;

-- Create masks for sensitive data
CREATE MASK credit_limit_mask ON CUSTOMER
FOR COLUMN CREDIT_LIMIT
RETURN
  CASE
    WHEN VERIFY_GROUP_FOR_USER(SESSION_USER, 'FINANCE') = 1 
      OR VERIFY_GROUP_FOR_USER(SESSION_USER, 'ADMIN') = 1
      OR SESSION_USER = SALES_REP_ID
      THEN CREDIT_LIMIT
    ELSE NULL
  END
ENABLE;

CREATE MASK phone_mask ON CUSTOMER
FOR COLUMN PHONE
RETURN
  CASE
    WHEN VERIFY_GROUP_FOR_USER(SESSION_USER, 'CUSTOMER_SERVICE') = 1 
      OR VERIFY_GROUP_FOR_USER(SESSION_USER, 'ADMIN') = 1
      OR SESSION_USER = SALES_REP_ID
      THEN PHONE
    ELSE SUBSTR(PHONE, 1, LENGTH(PHONE) - 4) || 'XXXX'
  END
ENABLE;

CREATE MASK email_mask ON CUSTOMER
FOR COLUMN EMAIL
RETURN
  CASE
    WHEN VERIFY_GROUP_FOR_USER(SESSION_USER, 'MARKETING') = 1 
      OR VERIFY_GROUP_FOR_USER(SESSION_USER, 'ADMIN') = 1
      OR SESSION_USER = SALES_REP_ID
      THEN EMAIL
    ELSE REGEXP_REPLACE(EMAIL, '(.*)@', 'XXXX@')
  END
ENABLE;

-- Activate row and column access control
ALTER TABLE CUSTOMER
ACTIVATE ROW ACCESS CONTROL
ACTIVATE COLUMN ACCESS CONTROL;

-- Temporal Tables

-- Comprehensive System-Period Temporal Table Example
```sql
-- Create a system-period temporal table for tracking product changes
CREATE TABLE PRODUCT_HISTORY (
  PRODNO CHAR(10) NOT NULL,
  PRODNAME VARCHAR(100),
  CATEGORY VARCHAR(50),
  UNITPRICE DECIMAL(10,2),
  COST_PRICE DECIMAL(10,2),
  SUPPLIER_ID CHAR(10),
  STOCK_QUANTITY INTEGER,
  REORDER_LEVEL INTEGER,
  ACTIVE_FLAG CHAR(1),
  LAST_UPDATED_BY VARCHAR(50),
  SYS_START TIMESTAMP(12) NOT NULL GENERATED ALWAYS AS ROW BEGIN,
  SYS_END TIMESTAMP(12) NOT NULL GENERATED ALWAYS AS ROW END,
  TRANS_ID TIMESTAMP(12) GENERATED ALWAYS AS TRANSACTION START ID,
  PERIOD SYSTEM_TIME (SYS_START, SYS_END)
) WITH SYSTEM VERSIONING;

-- Insert sample data
INSERT INTO PRODUCT_HISTORY (
  PRODNO, PRODNAME, CATEGORY, UNITPRICE, COST_PRICE, 
  SUPPLIER_ID, STOCK_QUANTITY, REORDER_LEVEL, ACTIVE_FLAG, LAST_UPDATED_BY
) VALUES (
  'P1001', 'Laptop Pro X1', 'ELECTRONICS', 1299.99, 950.00, 
  'S100', 25, 10, 'Y', 'SYSTEM'
);

-- Update the price - this will automatically create history
UPDATE PRODUCT_HISTORY
SET UNITPRICE = 1399.99,
    LAST_UPDATED_BY = 'PRICEADMIN'
WHERE PRODNO = 'P1001';

-- Query for current data
SELECT * FROM PRODUCT_HISTORY;

-- Query for historical data as of a specific date
SELECT * FROM PRODUCT_HISTORY
FOR SYSTEM_TIME AS OF '2023-01-15-00.00.00.000000';

-- Query for changes over a period
SELECT 
  p.PRODNO,
  p.PRODNAME,
  p.UNITPRICE,
  p.SYS_START AS EFFECTIVE_FROM,
  p.SYS_END AS EFFECTIVE_TO,
  p.LAST_UPDATED_BY,
  CASE 
    WHEN p.SYS_END = '9999-12-30-00.00.00.000000' THEN 'Current'
    ELSE 'Historical'
  END AS RECORD_STATUS
FROM PRODUCT_HISTORY FOR SYSTEM_TIME ALL p
WHERE p.PRODNO = 'P1001'
ORDER BY p.SYS_START;

