﻿-- Performance Optimization
SET SCHEMA DEMOSCHEMA;

-- Query Rewrite for JOIN Optimization

-- Original simple query
SELECT 
  c.CUSTNAME,
  SUM(o.TOTAL_AMOUNT) AS TOTAL_ORDERS
FROM CUSTOMER c
LEFT JOIN ORDERS o ON c.CUSTNO = o.CUSTNO
WHERE (o.ORDERDATE >= '2023-01-01' OR o.ORDERDATE IS NULL)
GROUP BY c.CUSTNAME
ORDER BY TOTAL_ORDERS DESC;

-- Rewritten simple query

SELECT 
  c.CUSTNAME,
  COALESCE(o.TOTAL_ORDERS, 0) AS TOTAL_ORDERS
FROM CUSTOMER c
LEFT JOIN (
  SELECT 
    CUSTNO, 
    SUM(TOTAL_AMOUNT) AS TOTAL_ORDERS
  FROM ORDERS
  WHERE ORDERDATE >= '2023-01-01'
  GROUP BY CUSTNO
) o ON c.CUSTNO = o.CUSTNO
ORDER BY TOTAL_ORDERS DESC;


-- Original complex query with multiple joins
SELECT 
  c.CUSTNAME,
  p.PRODNAME,
  SUM(ol.QUANTITY) AS TOTAL_QUANTITY,
  SUM(ol.QUANTITY * p.UNITPRICE) AS TOTAL_AMOUNT
FROM CUSTOMER c
JOIN ORDERS o ON c.CUSTNO = o.CUSTNO
JOIN ORDERLINE ol ON o.ORDERNO = ol.ORDERNO
JOIN PRODUCT p ON ol.PRODNO = p.PRODNO
WHERE o.ORDERDATE BETWEEN '2023-01-01' AND '2023-12-31'
GROUP BY c.CUSTNAME, p.PRODNAME
ORDER BY c.CUSTNAME, p.PRODNAME;

-- Optimized query using intermediate aggregation
WITH order_totals AS (
  SELECT 
    o.CUSTNO,
    ol.PRODNO,
    SUM(ol.QUANTITY) AS TOTAL_QUANTITY,
    SUM(ol.QUANTITY * p.UNITPRICE) AS TOTAL_AMOUNT
  FROM ORDERS o
  JOIN ORDERLINE ol ON o.ORDERNO = ol.ORDERNO
  JOIN PRODUCT p ON ol.PRODNO = p.PRODNO
  WHERE o.ORDERDATE BETWEEN '2023-01-01' AND '2023-12-31'
  GROUP BY o.CUSTNO, ol.PRODNO
)
SELECT 
  c.CUSTNAME,
  p.PRODNAME,
  ot.TOTAL_QUANTITY,
  ot.TOTAL_AMOUNT
FROM order_totals ot
JOIN CUSTOMER c ON ot.CUSTNO = c.CUSTNO
JOIN PRODUCT p ON ot.PRODNO = p.PRODNO
ORDER BY c.CUSTNAME, p.PRODNAME;


-- GROUPING SETS for Complex Aggregation
SELECT 
  COALESCE(YEAR_VAL, 0) AS ORDER_YEAR,
  COALESCE(MONTH_VAL, 0) AS ORDER_MONTH,
  COALESCE(CATEGORY, 'ALL') AS CATEGORY,
  SUM(QUANTITY * UNITPRICE) AS TOTAL_SALES,
  GROUPING(YEAR_VAL) AS GRP_YEAR,
  GROUPING(MONTH_VAL) AS GRP_MONTH,
  GROUPING(CATEGORY) AS GRP_PRODLINE
FROM (
  SELECT 
    YEAR(o.ORDERDATE) AS YEAR_VAL,
    MONTH(o.ORDERDATE) AS MONTH_VAL,
    p.CATEGORY,
    ol.QUANTITY,
    ol.UNIT_PRICE AS UNITPRICE
  FROM ORDERS o
  JOIN ORDERLINE ol ON o.ORDERNO = ol.ORDERNO
  JOIN PRODUCT p ON ol.PRODNO = p.PRODNO
) AS derived_data
GROUP BY GROUPING SETS (
  (YEAR_VAL, MONTH_VAL, CATEGORY),
  (YEAR_VAL, CATEGORY),
  (YEAR_VAL),
  ()
)
ORDER BY GRP_YEAR, ORDER_YEAR, GRP_MONTH, ORDER_MONTH, GRP_PRODLINE, CATEGORY;

-- Using GROUPING SETS for multi-dimensional analysis
WITH order_data AS (
  SELECT 
    YEAR(o.ORDERDATE) AS year_val,
    QUARTER(o.ORDERDATE) AS quarter_val,
    p.CATEGORY,
    c.STATE,
    o.ORDERNO,
    o.CUSTNO,
    ol.QUANTITY * p.UNITPRICE AS line_total
  FROM ORDERS o
  JOIN ORDERLINE ol ON o.ORDERNO = ol.ORDERNO
  JOIN PRODUCT p ON ol.PRODNO = p.PRODNO
  JOIN CUSTOMER c ON o.CUSTNO = c.CUSTNO
  WHERE YEAR(o.ORDERDATE) = 2023
)
SELECT 
  COALESCE(year_val, 0) AS ORDER_YEAR,
  COALESCE(quarter_val, 0) AS ORDER_QUARTER,
  COALESCE(CATEGORY, 'All Categories') AS CATEGORY,
  COALESCE(STATE, 'All States') AS STATE,
  SUM(line_total) AS TOTAL_SALES,
  COUNT(DISTINCT ORDERNO) AS ORDER_COUNT,
  COUNT(DISTINCT CUSTNO) AS CUSTOMER_COUNT,
  GROUPING(year_val) AS GRP_YEAR,
  GROUPING(quarter_val) AS GRP_QUARTER,
  GROUPING(CATEGORY) AS GRP_CATEGORY,
  GROUPING(STATE) AS GRP_STATE
FROM order_data
GROUP BY GROUPING SETS (
  (year_val, quarter_val, CATEGORY, STATE),
  (year_val, quarter_val, CATEGORY),
  (year_val, quarter_val, STATE),
  (year_val, CATEGORY, STATE),
  (year_val, CATEGORY),
  (year_val, STATE),
  (year_val),
  ()
)
ORDER BY 
  GRP_YEAR, ORDER_YEAR,
  GRP_QUARTER, ORDER_QUARTER,
  GRP_CATEGORY, CATEGORY,
  GRP_STATE, STATE;
 

-- Materialized Query Table (MQT)

DROP TABLE monthly_product_sales_mqt;

-- Create a materialized query table for sales analysis
CREATE TABLE monthly_product_sales_mqt AS (
  SELECT 
    YEAR(o.ORDERDATE) AS SALES_YEAR,
    MONTH(o.ORDERDATE) AS SALES_MONTH,
    p.CATEGORY,
    p.PRODNO,
    p.PRODNAME,
    SUM(ol.QUANTITY) AS TOTAL_QUANTITY,
    SUM(ol.QUANTITY * p.UNITPRICE) AS TOTAL_SALES,
    AVG(p.UNITPRICE) AS AVG_PRICE,
    COUNT(DISTINCT o.CUSTNO) AS CUSTOMER_COUNT,
    COUNT(DISTINCT o.ORDERNO) AS ORDER_COUNT
  FROM ORDERS o
  JOIN ORDERLINE ol ON o.ORDERNO = ol.ORDERNO
  JOIN PRODUCT p ON ol.PRODNO = p.PRODNO
  GROUP BY 
    YEAR(o.ORDERDATE), 
    MONTH(o.ORDERDATE), 
    p.CATEGORY,
    p.PRODNO,
    p.PRODNAME
) DATA INITIALLY IMMEDIATE REFRESH DEFERRED
MAINTAINED BY USER
ENABLE QUERY OPTIMIZATION;

-- Refresh the MQT
REFRESH TABLE monthly_product_sales_mqt;

-- Query using the MQT
SELECT 
  SALES_YEAR,
  SALES_MONTH,
  CATEGORY,
  SUM(TOTAL_SALES) AS CATEGORY_SALES,
  SUM(TOTAL_QUANTITY) AS CATEGORY_QUANTITY,
  SUM(CUSTOMER_COUNT) AS CUSTOMER_COUNT
FROM monthly_product_sales_mqt
WHERE SALES_YEAR = 2023
GROUP BY SALES_YEAR, SALES_MONTH, CATEGORY
ORDER BY SALES_YEAR, SALES_MONTH, CATEGORY;
 