﻿-- Advanced Indexing
SET SCHEMA DEMOSCHEMA;

-- Expression-Based Index for Case-Insensitive Search
-- Create an index for case-insensitive search
DROP INDEX ix_customer_upper_name;

CREATE  INDEX ix_customer_upper_name
ON CUSTOMER (UPPER(CUSTNAME));

-- Query that can use the expression index
SELECT * FROM CUSTOMER
WHERE UPPER(CUSTNAME) LIKE '%PART%';

-- Covering Index Example

-- Create a covering index for order queries with date ranges
DROP INDEX ix_orders_date_cust_amount;
CREATE INDEX ix_orders_date_cust_amount
ON ORDERS (ORDERDATE, CUSTNO, TOTAL_AMOUNT);

-- Query that can use the covering index
SELECT ORDERNO, ORDERDATE, CUSTNO, TOTAL_AMOUNT
FROM ORDERS
WHERE ORDERDATE BETWEEN '2023-01-01' AND '2023-03-31'
AND CUSTNO = 'C000000001' 
ORDER BY ORDERDATE DESC;
 

--- Index for JSON Data
-- Create an index on JSON property
-- WARNING THE JSON NEEDS TO BE STORED IN A VARCHAR NOT A CLOB !!
DROP INDEX ix_product_manufacturer;
CREATE INDEX ix_product_manufacturer
ON PRODUCT_CATALOG (JSON_VALUE(PRODUCT_DETAILS, '$.manufacturer'));

-- Query that can use the JSON index
SELECT 
  PRODUCT_ID, 
  PRODUCT_NAME, 
  JSON_VALUE(PRODUCT_DETAILS, '$.manufacturer') AS MANUFACTURER,
  JSON_VALUE(PRODUCT_DETAILS, '$.price') AS PRICE
FROM PRODUCT_CATALOG
WHERE JSON_VALUE(PRODUCT_DETAILS, '$.manufacturer') = 'Dell';


-- Finding unused indexes
SELECT 
  SYSTEM_TABLE_SCHEMA AS SCHEMA_NAME,
  SYSTEM_TABLE_NAME AS TABLE_NAME,
  SYSTEM_INDEX_NAME AS INDEX_NAME,
  LAST_QUERY_USE,
  LAST_STATISTICS_USE,
  DAYS(CURRENT DATE) - DAYS(COALESCE(LAST_QUERY_USE, LAST_STATISTICS_USE, CREATE_TIMESTAMP)) AS DAYS_UNUSED
FROM QSYS2.SYSINDEXSTAT
WHERE (LAST_QUERY_USE IS NULL OR DAYS(CURRENT DATE) - DAYS(LAST_QUERY_USE) > 90)
AND SYSTEM_TABLE_SCHEMA NOT IN ('QSYS', 'QSYS2')
ORDER BY DAYS_UNUSED DESC;

-- Creating an index with optimal column order (most selective first)
CREATE INDEX ix_customer_region_city ON CUSTOMER (
  STATUS,      -- Low cardinality (few distinct values)
  REGION,      -- Medium cardinality
  POSTALCODE,  -- High cardinality (many distinct values)
  CITY         -- Medium-high cardinality
);

-- Index status and usage information
SELECT *
FROM QSYS2.SYSINDEXES
WHERE SYSTEM_TABLE_SCHEMA = 'DEMOSCHEMA';
 

