﻿-- JSON Support
SET SCHEMA DEMOSCHEMA;
-- Creating JSON Objects
 
-- Create a complex JSON object from relational data
SELECT JSON_OBJECT(
  'customer' VALUE JSON_OBJECT(
    'id' VALUE c.CUSTNO,
    'name' VALUE c.CUSTNAME,
    'contactInfo' VALUE JSON_OBJECT(
      'email' VALUE c.EMAIL,
      'phone' VALUE c.PHONE,
      'address' VALUE JSON_OBJECT(
        'street' VALUE c.STREET,
        'city' VALUE c.CITY,
        'state' VALUE c.STATE,
        'zipCode' VALUE c.ZIPCODE,
        'country' VALUE c.COUNTRY
      )
    ),
    'status' VALUE c.STATUS,
    'creditLimit' VALUE c.CREDIT_LIMIT
  ),
  'orders' VALUE (
    SELECT JSON_ARRAYAGG(
      JSON_OBJECT(
        'orderNo' VALUE o.ORDERNO,
        'orderDate' VALUE o.ORDERDATE,
        'totalAmount' VALUE o.TOTAL_AMOUNT,
        'status' VALUE o.STATUS,
        'items' VALUE (
          SELECT JSON_ARRAYAGG(
            JSON_OBJECT(
              'productId' VALUE ol.PRODNO,
              'productName' VALUE p.PRODNAME,
              'quantity' VALUE ol.QUANTITY,
              'unitPrice' VALUE p.UNITPRICE,
              'lineTotal' VALUE ol.QUANTITY * p.UNITPRICE
            )
          )
          FROM ORDERLINE ol
          JOIN PRODUCT p ON ol.PRODNO = p.PRODNO
          WHERE ol.ORDERNO = o.ORDERNO
        )
      )
    )
    FROM ORDERS o
    WHERE o.CUSTNO = c.CUSTNO
    AND o.ORDERDATE >= CURRENT DATE - 1 YEAR
  )
) AS CUSTOMER_JSON
FROM CUSTOMER c
WHERE c.CUSTNO = 'C000000001';


-- Extracting Data with JSON_TABLE
-- JSON documents stored in a table
  SELECT ORDERNO, ORDER_JSON
  FROM JSON_ORDERS;
-- Extract order information from a JSON document
WITH order_json AS (
  SELECT ORDERNO, ORDER_JSON
  FROM JSON_ORDERS
  WHERE ORDERNO = 12345
)
SELECT 
  oj.ORDERNO,
  items.item_id,
  items.product_name,
  items.quantity,
  items.unit_price,
  items.line_total
FROM order_json oj,
JSON_TABLE(
  oj.ORDER_JSON,
  '$.items[*]' COLUMNS(
    item_id INT PATH '$.productId',
    product_name VARCHAR(100) PATH '$.productName',
    quantity INT PATH '$.quantity',
    unit_price DECIMAL(10,2) PATH '$.unitPrice',
    line_total DECIMAL(10,2) PATH '$.lineTotal'
  )
) AS items;
 

--JSON_VALUE for Filtering
SELECT * FROM  PRODUCT_CATALOG; 
-- Query products based on specific attributes in JSON
SELECT 
  PRODUCT_ID,
  PRODUCT_NAME,
  JSON_VALUE(PRODUCT_DETAILS, '$.manufacturer') AS MANUFACTURER,
  JSON_VALUE(PRODUCT_DETAILS, '$.specifications.weight') AS WEIGHT,
  JSON_VALUE(PRODUCT_DETAILS, '$.specifications.dimensions.height') AS HEIGHT,
  CAST(JSON_VALUE(PRODUCT_DETAILS, '$.price') AS DECIMAL(10,2)) AS PRICE
FROM PRODUCT_CATALOG
WHERE JSON_VALUE(PRODUCT_DETAILS, '$.manufacturer') = 'Apple'
AND CAST(JSON_VALUE(PRODUCT_DETAILS, '$.specifications.weight') AS DECIMAL(5,2)) < 1.0
ORDER BY CAST(JSON_VALUE(PRODUCT_DETAILS, '$.price') AS DECIMAL(10,2)) DESC;
 
 
-- Storing application configuration as JSON
CREATE OR REPLACE TABLE DEMOSCHEMA.APP_CONFIG (
  APP_ID VARCHAR(50) NOT NULL PRIMARY KEY,
  CONFIG_NAME VARCHAR(100) NOT NULL,
  CONFIG_DATA CLOB(2M),
  LAST_UPDATED TIMESTAMP DEFAULT CURRENT TIMESTAMP
);

-- Inserting configuration data
INSERT INTO APP_CONFIG (APP_ID, CONFIG_NAME, CONFIG_DATA)
VALUES ('INVENTORY_APP', 'System Settings', JSON_OBJECT(
  'theme' VALUE 'dark',
  'userPreferences' VALUE JSON_OBJECT(
    'defaultView' VALUE 'grid',
    'itemsPerPage' VALUE 50,
    'notifications' VALUE JSON_OBJECT(
      'email' VALUE 'true',
      'sms' VALUE 'false',
      'inApp' VALUE 'true'
    )
  ),
  'features' VALUE JSON_ARRAY('barcodeScanning', 'automaticReorder', 'reporting')
));

-- Retrieving a specific configuration value
SELECT JSON_VALUE(CONFIG_DATA, '$.userPreferences.itemsPerPage') AS ITEMS_PER_PAGE
FROM APP_CONFIG 
WHERE APP_ID = 'INVENTORY_APP' AND CONFIG_NAME = 'System Settings';
``` 
