﻿-- Window Functions
-- Basic Window Function Example
-- Calculate employee salary comparison within their department
SELECT 
  EMPNO,
  LASTNAME,
  DEPTNO,
  SALARY,
  AVG(SALARY) OVER (PARTITION BY DEPTNO) AS DEPT_AVG_SALARY,
  SALARY - AVG(SALARY) OVER (PARTITION BY DEPTNO) AS DIFF_FROM_AVG,
  DECIMAL((SALARY / DECIMAL(AVG(SALARY) OVER (PARTITION BY DEPTNO), 10, 2)) * 100 - 100, 5, 2) AS PERCENT_FROM_AVG,
  MIN(SALARY) OVER (PARTITION BY DEPTNO) AS DEPT_MIN_SALARY,
  MAX(SALARY) OVER (PARTITION BY DEPTNO) AS DEPT_MAX_SALARY
FROM EMPLOYEE
ORDER BY DEPTNO, SALARY DESC;

-- Ranking Functions Example
-- Compare different ranking functions
SELECT 
  EMPNO,
  LASTNAME,
  DEPTNO,
  SALARY,
  ROW_NUMBER() OVER (PARTITION BY DEPTNO ORDER BY SALARY DESC) AS DEPT_ROW_NUM,
  RANK() OVER (PARTITION BY DEPTNO ORDER BY SALARY DESC) AS DEPT_RANK,
  DENSE_RANK() OVER (PARTITION BY DEPTNO ORDER BY SALARY DESC) AS DEPT_DENSE_RANK,
  ROW_NUMBER() OVER (ORDER BY SALARY DESC) AS OVERALL_ROW_NUM,
  RANK() OVER (ORDER BY SALARY DESC) AS OVERALL_RANK,
  DENSE_RANK() OVER (ORDER BY SALARY DESC) AS OVERALL_DENSE_RANK
FROM EMPLOYEE
ORDER BY DEPTNO, SALARY DESC;
 
-- ROW_NUMBER vs. RANK vs. DENSE_RANK 
SELECT 
  EMPNO,
  LASTNAME,
  SALARY,
  ROW_NUMBER() OVER (ORDER BY SALARY DESC) AS ROW_NUM,
  RANK() OVER (ORDER BY SALARY DESC) AS RANK_NUM,
  DENSE_RANK() OVER (ORDER BY SALARY DESC) AS DENSE_RANK_NUM
FROM EMPLOYEE
ORDER BY SALARY DESC;


-- Navigation Functions Example
-- Basic example

SELECT 
  ORDERNO,
  ORDERDATE,
  CUSTNO,
  TOTAL_AMOUNT,
  LAG(TOTAL_AMOUNT) OVER (PARTITION BY CUSTNO ORDER BY ORDERDATE) AS PREV_ORDER_AMOUNT,
  TOTAL_AMOUNT - LAG(TOTAL_AMOUNT) OVER (PARTITION BY CUSTNO ORDER BY ORDERDATE) AS AMOUNT_CHANGE
FROM ORDERS
ORDER BY CUSTNO, ORDERDATE;

-- window frame specification

SELECT 
  EMPNO,
  LASTNAME,
  HIREDATE,
  SALARY,
  AVG(SALARY) OVER (
    ORDER BY HIREDATE 
    ROWS BETWEEN 1 PRECEDING AND 1 FOLLOWING
  ) AS ROLLING_AVG_SALARY
FROM EMPLOYEE
ORDER BY HIREDATE;

 
-- Analyze sales trends by comparing with previous and next periods
WITH MONTHLY_TOTALS AS (
  SELECT 
    YEAR(ORDERDATE) AS SALES_YEAR,
    MONTH(ORDERDATE) AS SALES_MONTH,
    SUM(TOTAL_AMOUNT) AS MONTHLY_SALES
  FROM ORDERS
  GROUP BY YEAR(ORDERDATE), MONTH(ORDERDATE)
)
SELECT 
  SALES_YEAR,
  SALES_MONTH,
  MONTHLY_SALES,
  LAG(MONTHLY_SALES, 1) OVER (ORDER BY SALES_YEAR, SALES_MONTH) AS PREV_MONTH_SALES,
  LEAD(MONTHLY_SALES, 1) OVER (ORDER BY SALES_YEAR, SALES_MONTH) AS NEXT_MONTH_SALES,
  MONTHLY_SALES - LAG(MONTHLY_SALES, 1) OVER (ORDER BY SALES_YEAR, SALES_MONTH) AS MONTH_OVER_MONTH_CHANGE,
  DECIMAL(
    (MONTHLY_SALES - LAG(MONTHLY_SALES, 1) OVER (ORDER BY SALES_YEAR, SALES_MONTH)) * 100.0 / 
    NULLIF(LAG(MONTHLY_SALES, 1) OVER (ORDER BY SALES_YEAR, SALES_MONTH), 0), 
    5, 2
  ) AS PERCENT_CHANGE,
  LAG(MONTHLY_SALES, 12) OVER (ORDER BY SALES_YEAR, SALES_MONTH) AS SAME_MONTH_PREV_YEAR,
  DECIMAL(
    (MONTHLY_SALES - LAG(MONTHLY_SALES, 12) OVER (ORDER BY SALES_YEAR, SALES_MONTH)) * 100.0 /
    NULLIF(LAG(MONTHLY_SALES, 12) OVER (ORDER BY SALES_YEAR, SALES_MONTH), 0),
    5, 2
  ) AS YEAR_OVER_YEAR_CHANGE
FROM MONTHLY_TOTALS
ORDER BY SALES_YEAR, SALES_MONTH;


-- Window Frame Specification Example
-- Calculate moving averages for sales analysis
SELECT 
  ORDERDATE,
  TOTAL_AMOUNT,
  -- 3-day moving average (including current day)
  AVG(TOTAL_AMOUNT) OVER (
    ORDER BY ORDERDATE 
    ROWS BETWEEN 2 PRECEDING AND CURRENT ROW
  ) AS MOVING_AVG_3DAY,
  -- 7-day moving average
  AVG(TOTAL_AMOUNT) OVER (
    ORDER BY ORDERDATE 
    ROWS BETWEEN 6 PRECEDING AND CURRENT ROW
  ) AS MOVING_AVG_7DAY,
  -- 30-day moving average using ROWS instead of RANGE
  AVG(TOTAL_AMOUNT) OVER (
    ORDER BY ORDERDATE 
    ROWS BETWEEN 29 PRECEDING AND CURRENT ROW
  ) AS MOVING_AVG_30DAY
FROM DAILY_SALES
WHERE ORDERDATE BETWEEN '2023-06-01' AND '2023-06-30'
ORDER BY ORDERDATE;
 

-- Complex Window Function for Percentiles
-- Calculate salary percentiles within departments
SELECT 
  EMPNO,
  LASTNAME,
  DEPTNO,
  SALARY,
  PERCENT_RANK() OVER (PARTITION BY DEPTNO ORDER BY SALARY) AS DEPT_PERCENT_RANK,
  CAST(PERCENT_RANK() OVER (PARTITION BY DEPTNO ORDER BY SALARY) * 100 AS DECIMAL(5,2)) AS DEPT_PERCENTILE,
  CUME_DIST() OVER (PARTITION BY DEPTNO ORDER BY SALARY) AS DEPT_CUME_DIST,
  CAST(CUME_DIST() OVER (PARTITION BY DEPTNO ORDER BY SALARY) * 100 AS DECIMAL(5,2)) AS DEPT_CUME_PERCENTILE,
  PERCENT_RANK() OVER (ORDER BY SALARY) AS OVERALL_PERCENT_RANK,
  CAST(PERCENT_RANK() OVER (ORDER BY SALARY) * 100 AS DECIMAL(5,2)) AS OVERALL_PERCENTILE
FROM EMPLOYEE
ORDER BY DEPTNO, SALARY;



-- Before optimization: Inefficient window function usage
SELECT 
  DEPTNO,
  EMPNO,
  LASTNAME,
  SALARY,
  AVG(SALARY) OVER (PARTITION BY DEPTNO) AS DEPT_AVG,
  MAX(SALARY) OVER (PARTITION BY DEPTNO) AS DEPT_MAX,
  MIN(SALARY) OVER (PARTITION BY DEPTNO) AS DEPT_MIN,
  SUM(SALARY) OVER (PARTITION BY DEPTNO) AS DEPT_SUM,
  COUNT(*) OVER (PARTITION BY DEPTNO) AS DEPT_COUNT
FROM EMPLOYEE;


-- After optimization: Combining window functions with same partitioning
WITH dept_stats AS (
  SELECT 
    DEPTNO,
    AVG(SALARY) AS DEPT_AVG,
    MAX(SALARY) AS DEPT_MAX,
    MIN(SALARY) AS DEPT_MIN,
    SUM(SALARY) AS DEPT_SUM,
    COUNT(*) AS DEPT_COUNT
  FROM EMPLOYEE
  GROUP BY DEPTNO
)
SELECT 
  e.DEPTNO,
  e.EMPNO,
  e.LASTNAME,
  e.SALARY,
  ds.DEPT_AVG,
  ds.DEPT_MAX,
  ds.DEPT_MIN,
  ds.DEPT_SUM,
  ds.DEPT_COUNT
FROM EMPLOYEE e
JOIN dept_stats ds ON e.DEPTNO = ds.DEPTNO;