﻿-- Advanced Joins
-- Complex Multi-Table Join Example
-- Complete sales analysis including product, customer, and regional information
SELECT 
  r.REGION_NAME,
  c.CUSTNAME,
  o.ORDERNO,
  o.ORDERDATE,
  p.PRODNAME,
  p.CATEGORY,
  ol.QUANTITY,
  p.UNITPRICE,
  ol.QUANTITY * p.UNITPRICE AS LINE_TOTAL,
  e.LASTNAME AS SALES_REP
FROM CUSTOMER c
JOIN REGION r ON c.REGION_ID = r.REGION_ID
JOIN ORDERS o ON c.CUSTNO = o.CUSTNO
JOIN ORDERLINE ol ON o.ORDERNO = ol.ORDERNO
JOIN PRODUCT p ON ol.PRODNO = p.PRODNO
JOIN EMPLOYEE e ON o.SALES_REP_ID = e.EMPNO
WHERE o.ORDERDATE BETWEEN '2023-01-01' AND '2023-12-31'
AND r.REGION_NAME IN ('Midwest', 'Northeast')
ORDER BY r.REGION_NAME, c.CUSTNAME, o.ORDERDATE;
    
-- Self-Join for Hierarchical Data
-- Employee and manager relationship with multiple organizational levels
SELECT 
  e.EMPNO,
  e.LASTNAME AS EMPLOYEE,
  e.JOB_TITLE,
  m1.LASTNAME AS MANAGER,
  m1.JOB_TITLE AS MANAGER_TITLE,
  m2.LASTNAME AS DIRECTOR,
  m2.JOB_TITLE AS DIRECTOR_TITLE,
  m3.LASTNAME AS VP,
  m3.JOB_TITLE AS VP_TITLE
FROM EMPLOYEE e
LEFT JOIN EMPLOYEE m1 ON e.MANAGERNO = m1.EMPNO -- Direct manager
LEFT JOIN EMPLOYEE m2 ON m1.MANAGERNO = m2.EMPNO -- Manager's manager (director)
LEFT JOIN EMPLOYEE m3 ON m2.MANAGERNO = m3.EMPNO -- Director's manager (VP)
ORDER BY m3.LASTNAME, m2.LASTNAME, m1.LASTNAME, e.LASTNAME;
; 

-- LATERAL JOIN Example
SELECT 
  d.DEPTNO, 
  d.DEPTNAME,
  e.EMPNO,
  e.LASTNAME
FROM DEPARTMENT d,
LATERAL (
  SELECT EMPNO, LASTNAME 
  FROM EMPLOYEE 
  WHERE DEPTNO = d.DEPTNO 
  ORDER BY SALARY DESC
  FETCH FIRST 3 ROWS ONLY
) AS e
ORDER BY d.DEPTNO;

-- Advanced Join with FULL OUTER JOIN
-- Compare budgeted versus actual expenses by department and expense category
SELECT 
  COALESCE(b.DEPTNO, a.DEPTNO) AS DEPTNO,
  COALESCE(b.EXPENSE_CATEGORY, a.EXPENSE_CATEGORY) AS EXPENSE_CATEGORY,
  b.BUDGETED_AMOUNT,
  a.ACTUAL_AMOUNT,
  a.ACTUAL_AMOUNT - b.BUDGETED_AMOUNT AS VARIANCE,
  CASE 
    WHEN b.BUDGETED_AMOUNT IS NULL THEN 'Unbudgeted Expense'
    WHEN a.ACTUAL_AMOUNT IS NULL THEN 'No Actual Expense'
    WHEN a.ACTUAL_AMOUNT > b.BUDGETED_AMOUNT THEN 'Over Budget'
    ELSE 'Under Budget'
  END AS STATUS
FROM BUDGET b
FULL OUTER JOIN ACTUAL_EXPENSES a 
  ON b.DEPTNO = a.DEPTNO 
  AND b.EXPENSE_CATEGORY = a.EXPENSE_CATEGORY
WHERE b.FISCAL_YEAR = 2023 AND a.FISCAL_YEAR = 2023
ORDER BY DEPTNO, EXPENSE_CATEGORY;


-- Advanced Subqueries
-- Correlated Subquery Example
-- Find employees with salaries higher than their department average
SELECT e.EMPNO, e.LASTNAME, e.DEPTNO, e.SALARY, (SELECT AVG(SALARY)
               FROM EMPLOYEE
               WHERE DEPTNO = e.DEPTNO) AS DEPT_AVG_SALARY, e.SALARY - (SELECT AVG(SALARY)
                   FROM EMPLOYEE
                   WHERE DEPTNO = e.DEPTNO) AS DIFF_FROM_AVG,
       DECIMAL((e.SALARY / (SELECT DECIMAL(AVG(SALARY), 10, 2)
                       FROM EMPLOYEE
                       WHERE DEPTNO = e.DEPTNO) * 100) - 100, 5, 2) AS PERCENT_ABOVE_AVG
    FROM EMPLOYEE e
    WHERE e.SALARY > (SELECT AVG(SALARY)
                FROM EMPLOYEE
                WHERE DEPTNO = e.DEPTNO)
    ORDER BY PERCENT_ABOVE_AVG DESC;


-- EXISTS vs. IN vs. JOIN Performance Comparison
-- Using EXISTS: Find customers who placed orders in 2024
SELECT c.CUSTNO, c.CUSTNAME
FROM CUSTOMER c
WHERE EXISTS (
  SELECT 1 FROM ORDERS o
  WHERE o.CUSTNO = c.CUSTNO
  AND o.ORDERDATE BETWEEN '2024-01-01' AND '2024-12-31'
);

-- Using IN: Same query with IN
SELECT c.CUSTNO, c.CUSTNAME
FROM CUSTOMER c
WHERE c.CUSTNO IN (
  SELECT o.CUSTNO FROM ORDERS o
  WHERE o.ORDERDATE BETWEEN '2024-01-01' AND '2024-12-31'
);

-- Using JOIN: Same query with JOIN
SELECT DISTINCT c.CUSTNO, c.CUSTNAME
FROM CUSTOMER c
JOIN ORDERS o ON c.CUSTNO = o.CUSTNO
WHERE o.ORDERDATE BETWEEN '2024-01-01' AND '2024-12-31';


-- Subquery with the ALL Operator
-- Find products with a price higher than all products in a specific category
SELECT 
  PRODNO, 
  PRODNAME, 
  CATEGORY, 
  UNITPRICE
FROM PRODUCT
WHERE UNITPRICE > ALL (
  SELECT UNITPRICE 
  FROM PRODUCT 
  WHERE CATEGORY = 'FURNITURE'
)
ORDER BY UNITPRICE;
 

-- Subquery with ANY/SOME
-- Find employees who earn more than any manager in the IT department
SELECT 
  EMPNO, 
  LASTNAME, 
  JOB_TITLE, 
  SALARY
FROM EMPLOYEE
WHERE SALARY > ANY (
  SELECT e.SALARY
  FROM EMPLOYEE e
  JOIN EMPLOYEE m ON e.MANAGERNO = m.EMPNO
  WHERE m.DEPTNO = 'D05'
)
ORDER BY SALARY;
 