﻿-- Common Table Expressions (CTEs)
SET SCHEMA DEMOSCHEMA;

-- SLIDE 12
-- Basic CTE Example
-- Find departments with more than 3 employees
WITH employee_counts AS (
  SELECT 
    DEPTNO,
    COUNT(*) AS EMP_COUNT 
  FROM EMPLOYEE 
  GROUP BY DEPTNO
)
SELECT 
  DEPTNO 
  EMP_COUNT
FROM employee_counts
WHERE EMP_COUNT > 3
ORDER BY EMP_COUNT DESC;
 
-- Multiple CTEs Example (Non recursive) -- SLIDE 14
-- Analyze departments by employee count and average salary
WITH 
dept_count AS (
  SELECT DEPTNO, COUNT(*) AS EMP_COUNT 
  FROM EMPLOYEE 
  GROUP BY DEPTNO
),
avg_salary AS (
  SELECT DEPTNO, AVG(SALARY) AS AVG_SAL 
  FROM EMPLOYEE 
  GROUP BY DEPTNO
),
salary_variance AS (
  SELECT 
    DEPTNO, 
    STDDEV(SALARY) AS SALARY_STDDEV
  FROM EMPLOYEE
  GROUP BY DEPTNO
)
SELECT 
  d.DEPTNAME, 
  dc.EMP_COUNT, 
  s.AVG_SAL,
  sv.SALARY_STDDEV,
  (s.AVG_SAL / dc.EMP_COUNT) AS AVG_SALARY_PER_EMPLOYEE
FROM DEPARTMENT d
JOIN dept_count dc ON d.DEPTNO = dc.DEPTNO
JOIN avg_salary s ON d.DEPTNO = s.DEPTNO
JOIN salary_variance sv ON d.DEPTNO = sv.DEPTNO
ORDER BY dc.EMP_COUNT DESC;
 
-- Recursive CTE Example - Organization Chart -- SLIDE 16
WITH RECURSIVE emp_hierarchy (EMPNO, FIRSTNAME, LASTNAME, MANAGERNO, LEVEL, ORG_PATH) AS (
  -- Base case: start with the CEO (employee with no manager)
  SELECT 
    EMPNO,
    FIRSTNAME,
    LASTNAME,
    MANAGERNO,
    1 AS LEVEL,
    CAST(LASTNAME || ', ' || FIRSTNAME AS VARCHAR(100)) AS ORG_PATH
  FROM EMPLOYEE
  WHERE MANAGERNO IS NULL
  
  UNION ALL
  
  -- Recursive case: get employees who report to already-retrieved managers
  SELECT 
    e.EMPNO,
    e.FIRSTNAME,
    e.LASTNAME,
    e.MANAGERNO,
    h.LEVEL + 1,
    CAST(h.ORG_PATH || ' > ' || e.LASTNAME || ', ' || e.FIRSTNAME AS VARCHAR(100)) AS ORG_PATH
  FROM EMPLOYEE e
  JOIN emp_hierarchy h ON e.MANAGERNO = h.EMPNO
)
SELECT 
  EMPNO,
  REPEAT('  ', LEVEL - 1) || FIRSTNAME || ' ' || LASTNAME AS EMPLOYEE,
  LEVEL,
  ORG_PATH
FROM emp_hierarchy
ORDER BY ORG_PATH;


-- CTE for Complex Bill of Materials Processing -- SLIDE 17
WITH RECURSIVE bom_explosion (PART_ID, PART_NAME, COMPONENT_ID, COMPONENT_QTY, TOTAL_QTY, LEVEL) AS (
  -- Base case: get top-level parts
  SELECT 
    p.PART_ID,
    p.PART_NAME,
    c.COMPONENT_ID,
    c.COMPONENT_QTY,
    c.COMPONENT_QTY AS TOTAL_QTY,
    1 AS LEVEL
  FROM PARTS p
  JOIN COMPONENTS c ON p.PART_ID = c.PART_ID
  WHERE p.PART_ID = 'A100' -- Starting with a specific part
  
  UNION ALL
  
  -- Recursive case: get sub-components
  SELECT 
    p.PART_ID,
    p.PART_NAME,
    c.COMPONENT_ID,
    c.COMPONENT_QTY,
    b.TOTAL_QTY * c.COMPONENT_QTY AS TOTAL_QTY,
    b.LEVEL + 1
  FROM bom_explosion b
  JOIN PARTS p ON b.COMPONENT_ID = p.PART_ID
  JOIN COMPONENTS c ON p.PART_ID = c.PART_ID
)
SELECT 
  PART_ID,
  PART_NAME,
  COMPONENT_ID,
  COMPONENT_QTY,
  TOTAL_QTY,
  LEVEL
FROM bom_explosion
ORDER BY LEVEL, PART_ID, COMPONENT_ID;

-- CTE approach -- SLIDE 18
WITH monthly_sales AS (
  SELECT 
    YEAR(ORDERDATE) AS YEAR,
    MONTH(ORDERDATE) AS MONTH,
    SUM(TOTAL_AMOUNT) AS SALES
  FROM ORDERS
  GROUP BY YEAR(ORDERDATE), MONTH(ORDERDATE)
)
SELECT * FROM monthly_sales ORDER BY YEAR, MONTH;

-- Temporary table approach -- SLIDE 18
DECLARE GLOBAL TEMPORARY TABLE session.monthly_sales AS (
  SELECT 
    YEAR(ORDERDATE) AS YEAR,
    MONTH(ORDERDATE) AS MONTH,
    SUM(TOTAL_AMOUNT) AS SALES
  FROM ORDERS
  GROUP BY YEAR(ORDERDATE), MONTH(ORDERDATE)
) WITH DATA;

SELECT * FROM session.monthly_sales ORDER BY YEAR, MONTH;
