﻿-- DB2 for i Schema Definition and Sample Data
-- This script creates all the tables used in the presentation examples
-- and populates them with sample data.

-- Create schema
CREATE SCHEMA DEMOSCHEMA;
SET SCHEMA DEMOSCHEMA;

-- DEPARTMENT table
CREATE OR REPLACE TABLE DEPARTMENT (
  DEPTNO CHAR(3) NOT NULL PRIMARY KEY,
  DEPTNAME VARCHAR(30) NOT NULL,
  MANAGER CHAR(6),
  LOCATION VARCHAR(30),
  DIVISION VARCHAR(30)
);

 
-- EMPLOYEE table
CREATE OR REPLACE TABLE EMPLOYEE (
  EMPNO CHAR(6) NOT NULL PRIMARY KEY,
  FIRSTNAME VARCHAR(30) NOT NULL,
  LASTNAME VARCHAR(30) NOT NULL,
  DEPTNO CHAR(3) NOT NULL,
  JOB_TITLE VARCHAR(50),
  SALARY DECIMAL(9,2),
  BONUS DECIMAL(9,2),
  COMM DECIMAL(9,2),
  HIREDATE DATE,
  BIRTHDATE DATE,
  MANAGERNO CHAR(6),
  EMAIL VARCHAR(100),
  PHONE VARCHAR(15),
  CONSTRAINT FK_EMP_DEPT FOREIGN KEY (DEPTNO) REFERENCES DEPARTMENT(DEPTNO)
);

-- Add self-referencing foreign key for manager relationship
ALTER TABLE DEPARTMENT 
  ADD CONSTRAINT FK_DEPT_MGR FOREIGN KEY (MANAGER) REFERENCES EMPLOYEE(EMPNO);

-- REGION table
CREATE OR REPLACE TABLE REGION (
  REGION_ID CHAR(2) NOT NULL PRIMARY KEY,
  REGION_NAME VARCHAR(30) NOT NULL,
  SALES_TARGET DECIMAL(12,2)
);

-- CUSTOMER table
CREATE OR REPLACE TABLE CUSTOMER (
  CUSTNO CHAR(10) NOT NULL PRIMARY KEY,
  CUSTNAME VARCHAR(50) NOT NULL,
  STREET VARCHAR(100),
  CITY VARCHAR(50),
  STATE CHAR(2),
  ZIPCODE VARCHAR(10),
  COUNTRY VARCHAR(50) DEFAULT 'USA',
  PHONE VARCHAR(15),
  EMAIL VARCHAR(100),
  CREDIT_LIMIT DECIMAL(10,2),
  DISCOUNT_PCT DECIMAL(4,2) DEFAULT 0,
  STATUS VARCHAR(10) DEFAULT 'ACTIVE',
  REGION_ID CHAR(2),
  ACTIVITY_LEVEL VARCHAR(10),
  SALES_REP_ID CHAR(6),
  DISCOUNT_TIER VARCHAR(10) DEFAULT 'STANDARD',
  CONSTRAINT FK_CUST_REGION FOREIGN KEY (REGION_ID) REFERENCES REGION(REGION_ID),
  CONSTRAINT FK_CUST_SALESREP FOREIGN KEY (SALES_REP_ID) REFERENCES EMPLOYEE(EMPNO)
);

-- PRODUCT table
CREATE OR REPLACE TABLE PRODUCT (
  PRODNO CHAR(10) NOT NULL PRIMARY KEY,
  PRODNAME VARCHAR(100) NOT NULL,
  CATEGORY VARCHAR(50),
  UNITPRICE DECIMAL(10,2) NOT NULL,
  COST_PRICE DECIMAL(10,2),
  SUPPLIER_ID CHAR(10),
  STOCK_QUANTITY INTEGER DEFAULT 0,
  REORDER_LEVEL INTEGER DEFAULT 10,
  ACTIVE_FLAG CHAR(1) DEFAULT 'Y'
);

-- ORDERS table
CREATE OR REPLACE TABLE ORDERS (
  ORDERNO INTEGER NOT NULL PRIMARY KEY,
  ORDERDATE DATE NOT NULL,
  CUSTNO CHAR(10) NOT NULL,
  TOTAL_AMOUNT DECIMAL(12,2),
  SALES_REP_ID CHAR(6),
  STATUS VARCHAR(20) DEFAULT 'COMPLETED',
  SHIP_DATE DATE,
  CONSTRAINT FK_ORD_CUST FOREIGN KEY (CUSTNO) REFERENCES CUSTOMER(CUSTNO),
  CONSTRAINT FK_ORD_SALESREP FOREIGN KEY (SALES_REP_ID) REFERENCES EMPLOYEE(EMPNO)
);

-- ORDERLINE table
CREATE OR REPLACE TABLE ORDERLINE (
  ORDERNO INTEGER NOT NULL,
  LINENO INTEGER NOT NULL,
  PRODNO CHAR(10) NOT NULL,
  QUANTITY INTEGER NOT NULL,
  UNIT_PRICE DECIMAL(10,2),
  PRIMARY KEY (ORDERNO, LINENO),
  CONSTRAINT FK_ORDLINE_ORD FOREIGN KEY (ORDERNO) REFERENCES ORDERS(ORDERNO) ON DELETE CASCADE,
  CONSTRAINT FK_ORDLINE_PROD FOREIGN KEY (PRODNO) REFERENCES PRODUCT(PRODNO)
);

-- DAILY_SALES table for time series analysis
CREATE OR REPLACE TABLE DAILY_SALES (
  ORDERDATE DATE NOT NULL PRIMARY KEY,
  TOTAL_AMOUNT DECIMAL(12,2),
  ORDER_COUNT INTEGER,
  CUSTOMER_COUNT INTEGER
);

-- APP_USERS table for security examples
CREATE OR REPLACE TABLE APP_USERS (
  USER_ID VARCHAR(50) NOT NULL PRIMARY KEY,
  USER_NAME VARCHAR(100),
  USER_ROLE VARCHAR(50),
  USER_REGION CHAR(2),
  CONSTRAINT FK_USER_REGION FOREIGN KEY (USER_REGION) REFERENCES REGION(REGION_ID)
);

-- BUDGET table for comparison examples
CREATE OR REPLACE TABLE BUDGET (
  DEPTNO CHAR(3) NOT NULL,
  EXPENSE_CATEGORY VARCHAR(50) NOT NULL,
  FISCAL_YEAR INTEGER NOT NULL,
  FISCAL_QUARTER INTEGER NOT NULL,
  BUDGETED_AMOUNT DECIMAL(12,2),
  PRIMARY KEY (DEPTNO, EXPENSE_CATEGORY, FISCAL_YEAR, FISCAL_QUARTER),
  CONSTRAINT FK_BUDGET_DEPT FOREIGN KEY (DEPTNO) REFERENCES DEPARTMENT(DEPTNO)
);

-- ACTUAL_EXPENSES table for comparison examples
CREATE OR REPLACE TABLE ACTUAL_EXPENSES (
  DEPTNO CHAR(3) NOT NULL,
  EXPENSE_CATEGORY VARCHAR(50) NOT NULL,
  FISCAL_YEAR INTEGER NOT NULL,
  FISCAL_QUARTER INTEGER NOT NULL,
  ACTUAL_AMOUNT DECIMAL(12,2),
  PRIMARY KEY (DEPTNO, EXPENSE_CATEGORY, FISCAL_YEAR, FISCAL_QUARTER),
  CONSTRAINT FK_ACTUAL_DEPT FOREIGN KEY (DEPTNO) REFERENCES DEPARTMENT(DEPTNO)
);

-- PARTS table for BOM examples
CREATE OR REPLACE TABLE PARTS (
  PART_ID CHAR(10) NOT NULL PRIMARY KEY,
  PART_NAME VARCHAR(100) NOT NULL,
  PART_TYPE VARCHAR(20),
  UNIT_COST DECIMAL(10,2)
);

-- COMPONENTS table for BOM examples
CREATE OR REPLACE TABLE COMPONENTS (
  PART_ID CHAR(10) NOT NULL,
  COMPONENT_ID CHAR(10) NOT NULL,
  COMPONENT_QTY INTEGER NOT NULL,
  PRIMARY KEY (PART_ID, COMPONENT_ID),
  CONSTRAINT FK_COMP_PART FOREIGN KEY (PART_ID) REFERENCES PARTS(PART_ID),
  CONSTRAINT FK_COMP_COMP FOREIGN KEY (COMPONENT_ID) REFERENCES PARTS(PART_ID)
);

-- CUSTOMER_STATUS table for merge examples
CREATE OR REPLACE TABLE CUSTOMER_STATUS (
  CUSTNO CHAR(10) NOT NULL PRIMARY KEY,
  CUSTNAME VARCHAR(50),
  LAST_ORDER_DATE DATE,
  ANNUAL_SPEND DECIMAL(12,2),
  STATUS VARCHAR(20),
  LAST_UPDATED TIMESTAMP,
  CONSTRAINT FK_CUSTSTAT_CUST FOREIGN KEY (CUSTNO) REFERENCES CUSTOMER(CUSTNO)
);

-- JSON_ORDERS table for JSON examples
CREATE OR REPLACE TABLE JSON_ORDERS (
  ORDERNO INTEGER NOT NULL PRIMARY KEY,
  ORDER_JSON CLOB(2M),
  CREATED_AT TIMESTAMP DEFAULT CURRENT TIMESTAMP
);

-- PRODUCT_CATALOG table for JSON examples
CREATE OR REPLACE TABLE PRODUCT_CATALOG (
  PRODUCT_ID CHAR(10) NOT NULL PRIMARY KEY,
  PRODUCT_NAME VARCHAR(100) NOT NULL,
  PRODUCT_DETAILS CLOB(2M),
  CREATED_AT TIMESTAMP DEFAULT CURRENT TIMESTAMP,
  LAST_UPDATED TIMESTAMP DEFAULT CURRENT TIMESTAMP
);


-- Table for system versioning (temporal) example
ALTER TABLE EMPLOYEE
 ADD COLUMN row_birth TIMESTAMP(12) NOT NULL IMPLICITLY HIDDEN GENERATED ALWAYS AS ROW BEGIN
 ADD COLUMN row_death  TIMESTAMP(12) NOT NULL IMPLICITLY HIDDEN 
   GENERATED ALWAYS AS ROW END    
 ADD COLUMN transaction_time 
   TIMESTAMP(12) IMPLICITLY HIDDEN GENERATED ALWAYS AS TRANSACTION START ID
 ADD PERIOD SYSTEM_TIME (row_birth, row_death);

 
CREATE TABLE EMPLOYEE_HISTORY LIKE EMPLOYEE;

ALTER TABLE EMPLOYEE 
   ADD VERSIONING USE HISTORY TABLE EMPLOYEE_HISTORY;
   
-- Populate tables with sample data
----------------------------------

-- Insert REGION data
INSERT INTO REGION (REGION_ID, REGION_NAME, SALES_TARGET) VALUES
('NE', 'Northeast', 2500000.00),
('MW', 'Midwest', 1750000.00),
('SO', 'South', 2000000.00),
('WE', 'West', 3000000.00),
('NW', 'Northwest', 1250000.00),
('SW', 'Southwest', 1500000.00);

-- Insert DEPARTMENT data
INSERT INTO DEPARTMENT (DEPTNO, DEPTNAME, LOCATION, DIVISION) VALUES
('D01', 'Executive', 'Chicago', 'Corporate'),
('D02', 'Finance', 'Chicago', 'Corporate'),
('D03', 'Marketing', 'New York', 'Sales'),
('D04', 'Sales', 'Dallas', 'Sales'),
('D05', 'IT', 'San Jose', 'Operations'),
('D06', 'Human Resources', 'Chicago', 'Corporate'),
('D07', 'Research', 'Boston', 'Product'),
('D08', 'Customer Service', 'Phoenix', 'Operations');

-- Insert EMPLOYEE data
INSERT INTO EMPLOYEE (EMPNO, FIRSTNAME, LASTNAME, DEPTNO, JOB_TITLE, SALARY, HIREDATE, BIRTHDATE, MANAGERNO, EMAIL, PHONE) VALUES
('E00001', 'John', 'Smith', 'D01', 'CEO', 250000.00, '2010-01-15', '1965-03-25', NULL, 'john.smith@company.com', '555-123-4567'),
('E00002', 'Sarah', 'Johnson', 'D02', 'CFO', 180000.00, '2011-03-20', '1970-07-12', 'E00001', 'sarah.johnson@company.com', '555-234-5678'),
('E00003', 'Robert', 'Williams', 'D03', 'Marketing Director', 150000.00, '2012-05-10', '1975-11-08', 'E00001', 'robert.williams@company.com', '555-345-6789'),
('E00004', 'Lisa', 'Brown', 'D04', 'Sales Director', 140000.00, '2012-08-15', '1972-04-18', 'E00001', 'lisa.brown@company.com', '555-456-7890'),
('E00005', 'David', 'Miller', 'D05', 'CTO', 175000.00, '2013-01-10', '1978-02-22', 'E00001', 'david.miller@company.com', '555-567-8901'),
('E00006', 'Jennifer', 'Davis', 'D06', 'HR Director', 125000.00, '2013-06-05', '1980-09-14', 'E00001', 'jennifer.davis@company.com', '555-678-9012'),
('E00007', 'Michael', 'Garcia', 'D07', 'Research Director', 155000.00, '2014-02-15', '1976-08-30', 'E00001', 'michael.garcia@company.com', '555-789-0123'),
('E00008', 'Patricia', 'Wilson', 'D08', 'Customer Service Director', 120000.00, '2014-07-20', '1982-05-17', 'E00001', 'patricia.wilson@company.com', '555-890-1234'),
('E00009', 'James', 'Taylor', 'D02', 'Financial Analyst', 85000.00, '2015-01-10', '1988-03-08', 'E00002', 'james.taylor@company.com', '555-901-2345'),
('E00010', 'Elizabeth', 'Anderson', 'D02', 'Accountant', 75000.00, '2015-04-15', '1990-11-22', 'E00002', 'elizabeth.anderson@company.com', '555-012-3456'),
('E00011', 'Thomas', 'Martinez', 'D03', 'Marketing Manager', 95000.00, '2015-08-20', '1985-07-04', 'E00003', 'thomas.martinez@company.com', '555-112-3344'),
('E00012', 'Nancy', 'Robinson', 'D03', 'Marketing Specialist', 65000.00, '2016-02-10', '1992-09-18', 'E00011', 'nancy.robinson@company.com', '555-223-4455'),
('E00013', 'Daniel', 'Clark', 'D04', 'Sales Manager', 90000.00, '2016-05-15', '1983-12-01', 'E00004', 'daniel.clark@company.com', '555-334-5566'),
('E00014', 'Karen', 'Rodriguez', 'D04', 'Sales Representative', 60000.00, '2016-09-10', '1991-05-28', 'E00013', 'karen.rodriguez@company.com', '555-445-6677'),
('E00015', 'Christopher', 'Lewis', 'D04', 'Sales Representative', 60000.00, '2017-01-20', '1990-02-14', 'E00013', 'christopher.lewis@company.com', '555-556-7788'),
('E00016', 'Laura', 'Lee', 'D05', 'IT Manager', 110000.00, '2017-04-15', '1984-10-09', 'E00005', 'laura.lee@company.com', '555-667-8899'),
('E00017', 'Steven', 'Walker', 'D05', 'Database Administrator', 95000.00, '2017-07-10', '1986-06-25', 'E00016', 'steven.walker@company.com', '555-778-9900'),
('E00018', 'Maria', 'Hall', 'D05', 'Systems Analyst', 85000.00, '2017-10-15', '1988-08-12', 'E00016', 'maria.hall@company.com', '555-889-0011'),
('E00019', 'Andrew', 'Allen', 'D06', 'HR Manager', 80000.00, '2018-01-15', '1987-03-30', 'E00006', 'andrew.allen@company.com', '555-990-1122'),
('E00020', 'Donna', 'Young', 'D07', 'Research Scientist', 95000.00, '2018-04-10', '1984-11-17', 'E00007', 'donna.young@company.com', '555-001-2233'),
('E00021', 'Richard', 'Hernandez', 'D07', 'Product Developer', 85000.00, '2018-07-15', '1986-12-23', 'E00007', 'richard.hernandez@company.com', '555-112-2334'),
('E00022', 'Kimberly', 'King', 'D08', 'Customer Service Manager', 70000.00, '2018-10-20', '1989-05-08', 'E00008', 'kimberly.king@company.com', '555-223-3445'),
('E00023', 'Charles', 'Wright', 'D08', 'Customer Service Rep', 50000.00, '2019-02-15', '1992-07-15', 'E00022', 'charles.wright@company.com', '555-334-4556'),
('E00024', 'Betty', 'Lopez', 'D08', 'Customer Service Rep', 50000.00, '2019-05-10', '1993-09-02', 'E00022', 'betty.lopez@company.com', '555-445-5667'),
('E00025', 'Donald', 'Scott', 'D04', 'Sales Representative', 62000.00, '2019-08-15', '1991-11-11', 'E00013', 'donald.scott@company.com', '555-556-6778');

-- Update DEPARTMENT with manager info
UPDATE DEPARTMENT
SET MANAGER = CASE DEPTNO
  WHEN 'D01' THEN 'E00001'
  WHEN 'D02' THEN 'E00002'
  WHEN 'D03' THEN 'E00003'
  WHEN 'D04' THEN 'E00004'
  WHEN 'D05' THEN 'E00005'
  WHEN 'D06' THEN 'E00006'
  WHEN 'D07' THEN 'E00007'
  WHEN 'D08' THEN 'E00008'
END;

-- Insert PRODUCT data
INSERT INTO PRODUCT (PRODNO, PRODNAME, CATEGORY, UNITPRICE, COST_PRICE, SUPPLIER_ID, STOCK_QUANTITY, REORDER_LEVEL) VALUES
('P0001', 'Laptop Pro', 'ELECTRONICS', 1299.99, 950.00, 'S001', 150, 20),
('P0002', 'Smartphone X', 'ELECTRONICS', 899.99, 650.00, 'S001', 200, 30),
('P0003', 'Wireless Headphones', 'ELECTRONICS', 199.99, 80.00, 'S002', 300, 50),
('P0004', 'Smart Watch', 'ELECTRONICS', 349.99, 150.00, 'S001', 100, 15),
('P0005', 'Tablet 10"', 'ELECTRONICS', 499.99, 300.00, 'S001', 120, 20),
('P0006', 'Office Chair', 'FURNITURE', 299.99, 150.00, 'S003', 50, 10),
('P0007', 'Desk Lamp', 'FURNITURE', 79.99, 30.00, 'S003', 80, 20),
('P0008', 'Ergonomic Desk', 'FURNITURE', 699.99, 400.00, 'S003', 30, 5),
('P0009', 'Wireless Keyboard', 'ELECTRONICS', 89.99, 40.00, 'S002', 150, 30),
('P0010', 'Wireless Mouse', 'ELECTRONICS', 59.99, 25.00, 'S002', 200, 40),
('P0011', 'External SSD 1TB', 'ELECTRONICS', 149.99, 80.00, 'S004', 120, 20),
('P0012', 'Monitor 27"', 'ELECTRONICS', 349.99, 200.00, 'S004', 80, 15),
('P0013', 'Laser Printer', 'ELECTRONICS', 399.99, 250.00, 'S004', 40, 10),
('P0014', 'Paper Shredder', 'OFFICE', 129.99, 70.00, 'S005', 30, 5),
('P0015', 'Filing Cabinet', 'FURNITURE', 199.99, 100.00, 'S003', 25, 5),
('P0016', 'Desk Organizer', 'OFFICE', 39.99, 15.00, 'S005', 100, 20),
('P0017', 'Whiteboard', 'OFFICE', 89.99, 40.00, 'S005', 40, 10),
('P0018', 'Conference Phone', 'ELECTRONICS', 299.99, 150.00, 'S002', 20, 5),
('P0019', 'Coffee Maker', 'APPLIANCES', 149.99, 75.00, 'S006', 35, 10),
('P0020', 'Microwave Oven', 'APPLIANCES', 199.99, 120.00, 'S006', 25, 5);

-- Insert CUSTOMER data
INSERT INTO CUSTOMER (CUSTNO, CUSTNAME, STREET, CITY, STATE, ZIPCODE, COUNTRY, PHONE, EMAIL, CREDIT_LIMIT, DISCOUNT_PCT, STATUS, REGION_ID, SALES_REP_ID) VALUES
('C000000001', 'Acme Corporation', '123 Main St', 'New York', 'NY', '10001', 'USA', '212-555-1234', 'contact@acme.com', 50000.00, 10.00, 'ACTIVE', 'NE', 'E00014'),
('C000000002', 'Global Industries', '456 Park Ave', 'Chicago', 'IL', '60601', 'USA', '312-555-5678', 'info@globalind.com', 75000.00, 15.00, 'ACTIVE', 'MW', 'E00014'),
('C000000003', 'Tech Solutions', '789 Market St', 'San Francisco', 'CA', '94103', 'USA', '415-555-9012', 'sales@techsolutions.com', 100000.00, 20.00, 'ACTIVE', 'WE', 'E00015'),
('C000000004', 'Data Systems Inc', '321 Commerce St', 'Dallas', 'TX', '75201', 'USA', '214-555-3456', 'info@datasystems.com', 60000.00, 12.00, 'ACTIVE', 'SO', 'E00013'),
('C000000005', 'Innovative Labs', '654 5th Ave', 'Seattle', 'WA', '98101', 'USA', '206-555-7890', 'contact@innovativelabs.com', 80000.00, 15.00, 'ACTIVE', 'NW', 'E00015'),
('C000000006', 'Metro Services', '987 Broad St', 'Boston', 'MA', '02110', 'USA', '617-555-2345', 'service@metroservices.com', 40000.00, 8.00, 'ACTIVE', 'NE', 'E00014'),
('C000000007', 'Pacific Partners', '741 Ocean Dr', 'Los Angeles', 'CA', '90001', 'USA', '213-555-6789', 'info@pacificpartners.com', 65000.00, 12.00, 'ACTIVE', 'WE', 'E00015'),
('C000000008', 'Mountain View Corp', '852 Summit Rd', 'Denver', 'CO', '80202', 'USA', '303-555-0123', 'sales@mountainview.com', 55000.00, 10.00, 'ACTIVE', 'SW', 'E00025'),
('C000000009', 'Sunrise Enterprises', '963 Dawn Ave', 'Miami', 'FL', '33101', 'USA', '305-555-4567', 'contact@sunriseent.com', 45000.00, 9.00, 'ACTIVE', 'SO', 'E00013'),
('C000000010', 'Horizon Group', '159 Skyline Blvd', 'Phoenix', 'AZ', '85001', 'USA', '602-555-8901', 'info@horizongroup.com', 70000.00, 14.00, 'ACTIVE', 'SW', 'E00025'),
('C000000011', 'Apex Systems', '753 Peak St', 'Atlanta', 'GA', '30301', 'USA', '404-555-2345', 'sales@apexsystems.com', 50000.00, 10.00, 'ACTIVE', 'SO', 'E00013'),
('C000000012', 'Coastal Solutions', '951 Harbor Dr', 'San Diego', 'CA', '92101', 'USA', '619-555-6789', 'info@coastalsolutions.com', 55000.00, 11.00, 'ACTIVE', 'WE', 'E00015'),
('C000000013', 'Central Corp', '357 Midway Rd', 'St. Louis', 'MO', '63101', 'USA', '314-555-0123', 'contact@centralcorp.com', 40000.00, 8.00, 'ACTIVE', 'MW', 'E00014'),
('C000000014', 'Sterling Partners', '246 Silver St', 'Detroit', 'MI', '48201', 'USA', '313-555-4567', 'info@sterlingpartners.com', 45000.00, 9.00, 'INACTIVE', 'MW', 'E00014'),
('C000000015', 'Evergreen Ltd', '135 Forest Ave', 'Portland', 'OR', '97201', 'USA', '503-555-8901', 'sales@evergreen.com', 50000.00, 10.00, 'ACTIVE', 'NW', 'E00015'),
('C000000016', 'Golden Gate Inc', '864 Bridge Rd', 'San Francisco', 'CA', '94105', 'USA', '415-555-2345', 'info@goldengate.com', 80000.00, 15.00, 'ACTIVE', 'WE', 'E00015'),
('C000000017', 'Liberty Group', '753 Freedom Dr', 'Philadelphia', 'PA', '19101', 'USA', '215-555-6789', 'contact@libertygroup.com', 60000.00, 12.00, 'ACTIVE', 'NE', 'E00014'),
('C000000018', 'Summit Corp', '951 Highland Ave', 'Minneapolis', 'MN', '55401', 'USA', '612-555-0123', 'info@summitcorp.com', 55000.00, 11.00, 'ACTIVE', 'MW', 'E00014'),
('C000000019', 'Ocean Systems', '246 Coastal Hwy', 'Miami', 'FL', '33101', 'USA', '305-555-4567', 'sales@oceansystems.com', 70000.00, 14.00, 'INACTIVE', 'SO', 'E00013'),
('C000000020', 'Alpine Technology', '864 Mountain Rd', 'Salt Lake City', 'UT', '84101', 'USA', '801-555-8901', 'info@alpinetech.com', 65000.00, 13.00, 'ACTIVE', 'SW', 'E00025');

-- Insert ORDERS and ORDERLINE data

-- First, clear any existing data if needed
DELETE FROM ORDERLINE;
DELETE FROM ORDERS;

-- Insert ORDERS data with valid customer IDs and dates
INSERT INTO ORDERS (ORDERNO, ORDERDATE, CUSTNO, TOTAL_AMOUNT, SALES_REP_ID, STATUS, SHIP_DATE) VALUES
(10001, DATE('2023-01-15'), 'C000000001', 2799.97, 'E00014', 'COMPLETED', DATE('2023-01-20')),
(10002, DATE('2023-01-18'), 'C000000002', 1599.98, 'E00014', 'COMPLETED', DATE('2023-01-25')),
(10003, DATE('2023-01-22'), 'C000000003', 3499.95, 'E00015', 'COMPLETED', DATE('2023-01-28')),
(10004, DATE('2023-02-05'), 'C000000001', 799.98, 'E00014', 'COMPLETED', DATE('2023-02-10')),
(10005, DATE('2023-02-12'), 'C000000004', 1699.97, 'E00013', 'COMPLETED', DATE('2023-02-18')),
(10006, DATE('2023-02-15'), 'C000000005', 2999.95, 'E00015', 'COMPLETED', DATE('2023-02-20')),
(10007, DATE('2023-02-20'), 'C000000006', 599.97, 'E00014', 'COMPLETED', DATE('2023-02-25')),
(10008, DATE('2023-02-28'), 'C000000007', 3999.96, 'E00015', 'COMPLETED', DATE('2023-03-05')),
(10009, DATE('2023-03-05'), 'C000000008', 1299.98, 'E00025', 'COMPLETED', DATE('2023-03-10')),
(10010, DATE('2023-03-10'), 'C000000009', 1999.97, 'E00013', 'COMPLETED', DATE('2023-03-15')),
(10011, DATE('2023-03-15'), 'C000000010', 2599.98, 'E00025', 'COMPLETED', DATE('2023-03-20')),
(10012, DATE('2023-03-20'), 'C000000011', 1399.96, 'E00013', 'COMPLETED', DATE('2023-03-25')),
(10013, DATE('2023-03-28'), 'C000000012', 4499.95, 'E00015', 'COMPLETED', DATE('2023-04-02')),
(10014, DATE('2023-04-05'), 'C000000013', 899.97, 'E00014', 'COMPLETED', DATE('2023-04-10')),
(10015, DATE('2023-04-10'), 'C000000014', 1799.98, 'E00014', 'COMPLETED', DATE('2023-04-15')),
(10016, DATE('2023-04-15'), 'C000000015', 2699.96, 'E00015', 'COMPLETED', DATE('2023-04-20')),
(10017, DATE('2023-04-22'), 'C000000016', 5999.95, 'E00015', 'COMPLETED', DATE('2023-04-28')),
(10018, DATE('2023-04-28'), 'C000000017', 999.98, 'E00014', 'COMPLETED', DATE('2023-05-03')),
(10019, DATE('2023-05-03'), 'C000000018', 1599.97, 'E00014', 'COMPLETED', DATE('2023-05-08')),
(10020, DATE('2023-05-10'), 'C000000020', 3799.96, 'E00025', 'COMPLETED', DATE('2023-05-15')),
(10021, DATE('2023-05-15'), 'C000000001', 1099.99, 'E00014', 'COMPLETED', DATE('2023-05-20')),
(10022, DATE('2023-05-22'), 'C000000002', 2199.98, 'E00014', 'COMPLETED', DATE('2023-05-27')),
(10023, DATE('2023-05-28'), 'C000000003', 4299.97, 'E00015', 'COMPLETED', DATE('2023-06-02')),
(10024, DATE('2023-06-03'), 'C000000004', 1899.96, 'E00013', 'COMPLETED', DATE('2023-06-08')),
(10025, DATE('2023-06-10'), 'C000000005', 3699.95, 'E00015', 'COMPLETED', DATE('2023-06-15')),
(10026, DATE('2023-06-15'), 'C000000006', 799.98, 'E00014', 'COMPLETED', DATE('2023-06-20')),
(10027, DATE('2023-06-22'), 'C000000007', 2999.97, 'E00015', 'COMPLETED', DATE('2023-06-27')),
(10028, DATE('2023-06-28'), 'C000000008', 1399.96, 'E00025', 'COMPLETED', DATE('2023-07-03')),
(10029, DATE('2023-07-05'), 'C000000009', 2599.95, 'E00013', 'COMPLETED', DATE('2023-07-10')),
(10030, DATE('2023-07-10'), 'C000000010', 3999.94, 'E00025', 'COMPLETED', DATE('2023-07-15')),
(10031, DATE('2023-07-15'), 'C000000011', 899.98, 'E00013', 'COMPLETED', DATE('2023-07-20')),
(10032, DATE('2023-07-22'), 'C000000012', 1799.97, 'E00015', 'COMPLETED', DATE('2023-07-27')),
(10033, DATE('2023-07-28'), 'C000000013', 3999.96, 'E00014', 'COMPLETED', DATE('2023-08-02')),
(10034, DATE('2023-08-03'), 'C000000014', 1599.95, 'E00014', 'COMPLETED', DATE('2023-08-08')),
(10035, DATE('2023-08-10'), 'C000000015', 2299.94, 'E00015', 'COMPLETED', DATE('2023-08-15')),
(10036, DATE('2023-08-15'), 'C000000016', 4499.93, 'E00015', 'COMPLETED', DATE('2023-08-20')),
(10037, DATE('2023-08-22'), 'C000000017', 1299.92, 'E00014', 'COMPLETED', DATE('2023-08-27')),
(10038, DATE('2023-08-28'), 'C000000018', 3599.91, 'E00014', 'COMPLETED', DATE('2023-09-02')),
(10039, DATE('2023-09-05'), 'C000000020', 2499.90, 'E00025', 'COMPLETED', DATE('2023-09-10')),
(10040, DATE('2023-09-10'), 'C000000001', 3699.89, 'E00014', 'COMPLETED', DATE('2023-09-15')),
(10041, DATE('2023-09-15'), 'C000000002', 999.99, 'E00014', 'COMPLETED', DATE('2023-09-20')),
(10042, DATE('2023-09-22'), 'C000000003', 2499.98, 'E00015', 'COMPLETED', DATE('2023-09-27')),
(10043, DATE('2023-09-28'), 'C000000004', 5999.97, 'E00013', 'COMPLETED', DATE('2023-10-03')),
(10044, DATE('2023-10-05'), 'C000000005', 1299.96, 'E00015', 'COMPLETED', DATE('2023-10-10')),
(10045, DATE('2023-10-10'), 'C000000006', 2599.95, 'E00014', 'COMPLETED', DATE('2023-10-15')),
(10046, DATE('2023-10-15'), 'C000000007', 3799.94, 'E00015', 'COMPLETED', DATE('2023-10-20')),
(10047, DATE('2023-10-22'), 'C000000008', 999.93, 'E00025', 'COMPLETED', DATE('2023-10-27')),
(10048, DATE('2023-10-28'), 'C000000009', 4599.92, 'E00013', 'COMPLETED', DATE('2023-11-02')),
(10049, DATE('2023-11-05'), 'C000000010', 1899.91, 'E00025', 'COMPLETED', DATE('2023-11-10')),
(10050, DATE('2023-11-10'), 'C000000011', 3299.90, 'E00013', 'COMPLETED', DATE('2023-11-15')),
(10051, DATE('2023-11-15'), 'C000000012', 6599.89, 'E00015', 'COMPLETED', DATE('2023-11-20')),
(10052, DATE('2023-11-22'), 'C000000013', 1299.88, 'E00014', 'COMPLETED', DATE('2023-11-27')),
(10053, DATE('2023-11-28'), 'C000000014', 2599.87, 'E00014', 'COMPLETED', DATE('2023-12-03')),
(10054, DATE('2023-12-05'), 'C000000015', 5999.86, 'E00015', 'COMPLETED', DATE('2023-12-10')),
(10055, DATE('2023-12-10'), 'C000000016', 1299.85, 'E00015', 'COMPLETED', DATE('2023-12-15')),
(10056, DATE('2023-12-15'), 'C000000017', 4599.84, 'E00014', 'COMPLETED', DATE('2023-12-20')),
(10057, DATE('2023-12-22'), 'C000000018', 2399.83, 'E00014', 'COMPLETED', DATE('2023-12-27')),
(10058, DATE('2023-12-28'), 'C000000020', 3599.82, 'E00025', 'COMPLETED', DATE('2024-01-02')),
(10059, DATE('2024-01-05'), 'C000000001', 6999.81, 'E00014', 'COMPLETED', DATE('2024-01-10')),
(10060, DATE('2024-01-10'), 'C000000002', 2099.80, 'E00014', 'COMPLETED', DATE('2024-01-15')),
(10061, DATE('2024-01-15'), 'C000000003', 3499.79, 'E00015', 'COMPLETED', DATE('2024-01-20')),
(10062, DATE('2024-01-22'), 'C000000004', 4899.78, 'E00013', 'COMPLETED', DATE('2024-01-27')),
(10063, DATE('2024-01-28'), 'C000000005', 1399.77, 'E00015', 'COMPLETED', DATE('2024-02-02')),
(10064, DATE('2024-02-05'), 'C000000006', 2899.76, 'E00014', 'COMPLETED', DATE('2024-02-10')),
(10065, DATE('2024-02-10'), 'C000000007', 5499.75, 'E00015', 'COMPLETED', DATE('2024-02-15')),
(10066, DATE('2024-02-15'), 'C000000008', 1899.74, 'E00025', 'COMPLETED', DATE('2024-02-20')),
(10067, DATE('2024-02-22'), 'C000000009', 2899.73, 'E00013', 'COMPLETED', DATE('2024-02-27')),
(10068, DATE('2024-02-28'), 'C000000010', 4999.72, 'E00025', 'COMPLETED', DATE('2024-03-05')),
(10069, DATE('2024-03-05'), 'C000000011', 1499.71, 'E00013', 'COMPLETED', DATE('2024-03-10')),
(10070, DATE('2024-03-10'), 'C000000012', 3599.70, 'E00015', 'COMPLETED', DATE('2024-03-15')),
(10071, DATE('2024-03-15'), 'C000000013', 5999.69, 'E00014', 'COMPLETED', DATE('2024-03-18')),
(10072, DATE('2024-03-18'), 'C000000001', 2399.98, 'E00014', 'PROCESSING', NULL),
(10073, DATE('2024-03-19'), 'C000000003', 3599.97, 'E00015', 'PROCESSING', NULL),
(10074, DATE('2024-03-20'), 'C000000005', 1999.96, 'E00015', 'PROCESSING', NULL),
(10075, DATE('2024-03-21'), 'C000000007', 4599.95, 'E00015', 'PROCESSING', NULL);

-- Insert ORDERLINE data with valid order IDs and product IDs
INSERT INTO ORDERLINE (ORDERNO, LINENO, PRODNO, QUANTITY, UNIT_PRICE) VALUES
-- Order 10001
(10001, 1, 'P0001', 2, 1299.99),
(10001, 2, 'P0010', 2, 59.99),
(10001, 3, 'P0009', 2, 89.99),

-- Order 10002
(10002, 1, 'P0005', 3, 499.99),
(10002, 2, 'P0009', 1, 89.99),
(10002, 3, 'P0010', 1, 59.99),

-- Order 10003
(10003, 1, 'P0002', 3, 899.99),
(10003, 2, 'P0003', 4, 199.99),

-- Order 10004
(10004, 1, 'P0007', 5, 79.99),
(10004, 2, 'P0009', 5, 89.99),

-- Order 10005
(10005, 1, 'P0004', 3, 349.99),
(10005, 2, 'P0010', 2, 59.99),
(10005, 3, 'P0011', 3, 149.99),

-- Order 10006
(10006, 1, 'P0001', 2, 1299.99),
(10006, 2, 'P0004', 1, 349.99),
(10006, 3, 'P0006', 1, 299.99),

-- Order 10007
(10007, 1, 'P0010', 5, 59.99),
(10007, 2, 'P0009', 3, 89.99),

-- Order 10008
(10008, 1, 'P0001', 3, 1299.99),
(10008, 2, 'P0002', 1, 899.99),

-- Order 10009
(10009, 1, 'P0005', 2, 499.99),
(10009, 2, 'P0011', 2, 149.99),

-- Order 10010
(10010, 1, 'P0004', 3, 349.99),
(10010, 2, 'P0009', 5, 89.99),
(10010, 3, 'P0010', 3, 59.99),

-- Order 10011
(10011, 1, 'P0001', 2, 1299.99),

-- Order 10012
(10012, 1, 'P0011', 5, 149.99),
(10012, 2, 'P0007', 3, 79.99),
(10012, 3, 'P0016', 5, 39.99),

-- Order 10013
(10013, 1, 'P0008', 5, 699.99),
(10013, 2, 'P0011', 3, 149.99),
(10013, 3, 'P0009', 2, 89.99),

-- Order 10014
(10014, 1, 'P0007', 5, 79.99),
(10014, 2, 'P0016', 5, 39.99),
(10014, 3, 'P0017', 3, 89.99),

-- Order 10015
(10015, 1, 'P0005', 3, 499.99),
(10015, 2, 'P0011', 2, 149.99),

-- Adding more order lines for remaining orders
-- Order 10016
(10016, 1, 'P0001', 1, 1299.99),
(10016, 2, 'P0008', 2, 699.99),

-- Order 10017
(10017, 1, 'P0002', 5, 899.99),
(10017, 2, 'P0003', 5, 199.99),

-- Order 10018
(10018, 1, 'P0009', 6, 89.99),
(10018, 2, 'P0016', 5, 39.99),

-- Order 10019
(10019, 1, 'P0004', 3, 349.99),
(10019, 2, 'P0010', 5, 59.99),
(10019, 3, 'P0017', 2, 89.99),

-- Order 10020
(10020, 1, 'P0001', 2, 1299.99),
(10020, 2, 'P0002', 1, 899.99),
(10020, 3, 'P0009', 3, 89.99),

-- Continuing with more orders...
-- Order 10021
(10021, 1, 'P0003', 5, 199.99),
(10021, 2, 'P0007', 1, 79.99),

-- Order 10022
(10022, 1, 'P0001', 1, 1299.99),
(10022, 2, 'P0004', 2, 349.99),
(10022, 3, 'P0010', 1, 59.99),

-- Order 10023
(10023, 1, 'P0002', 3, 899.99),
(10023, 2, 'P0005', 1, 499.99),
(10023, 3, 'P0011', 3, 149.99),

-- Order 10024
(10024, 1, 'P0008', 2, 699.99),
(10024, 2, 'P0016', 4, 39.99),
(10024, 3, 'P0017', 1, 89.99),

-- Order 10025
(10025, 1, 'P0001', 2, 1299.99),
(10025, 2, 'P0004', 1, 349.99),
(10025, 3, 'P0011', 5, 149.99),

-- Let's continue with a few more orders
-- Order 10026
(10026, 1, 'P0007', 10, 79.99),

-- Order 10027
(10027, 1, 'P0001', 2, 1299.99),
(10027, 2, 'P0009', 2, 89.99),
(10027, 3, 'P0010', 1, 59.99),

-- Order 10028
(10028, 1, 'P0005', 2, 499.99),
(10028, 2, 'P0016', 10, 39.99),

-- Order 10029
(10029, 1, 'P0004', 5, 349.99),
(10029, 2, 'P0010', 5, 59.99),
(10029, 3, 'P0016', 5, 39.99),

-- Order 10030
(10030, 1, 'P0001', 3, 1299.99),
(10030, 2, 'P0011', 1, 149.99),

-- Order 10031
(10031, 1, 'P0007', 5, 79.99),
(10031, 2, 'P0010', 8, 59.99),

-- For the newest orders (processing status)
-- Order 10072
(10072, 1, 'P0001', 1, 1299.99),
(10072, 2, 'P0004', 2, 349.99),
(10072, 3, 'P0016', 10, 39.99),

-- Order 10073
(10073, 1, 'P0002', 4, 899.99),

-- Order 10074
(10074, 1, 'P0005', 4, 499.99),

-- Order 10075
(10075, 1, 'P0008', 5, 699.99),
(10075, 2, 'P0016', 5, 39.99),
(10075, 3, 'P0017', 10, 89.99);

-- Now update the total amounts in ORDERS based on the actual ORDERLINE data
UPDATE ORDERS o
SET TOTAL_AMOUNT = (
  SELECT SUM(QUANTITY * UNIT_PRICE)
  FROM ORDERLINE
  WHERE ORDERNO = o.ORDERNO
);

-- Verify the data
SELECT 'ORDERS populated successfully with ' || 
       CAST(COUNT(*) AS VARCHAR(10)) || ' orders' AS RESULT
FROM ORDERS;

SELECT 'ORDERLINE populated successfully with ' || 
       CAST(COUNT(*) AS VARCHAR(10)) || ' order lines' AS RESULT
FROM ORDERLINE;

-- Insert APP_USERS data for security examples
INSERT INTO APP_USERS (USER_ID, USER_NAME, USER_ROLE, USER_REGION) VALUES
('ADMIN', 'System Administrator', 'ADMIN', NULL),
('FINANCE1', 'Finance User', 'FINANCE', NULL),
('SALES1', 'East Sales User', 'SALES', 'NE'),
('SALES2', 'West Sales User', 'SALES', 'WE'),
('SALES3', 'South Sales User', 'SALES', 'SO'),
('MKTG1', 'Marketing User', 'MARKETING', NULL),
('CUST1', 'Customer Service User', 'CUSTOMER_SERVICE', NULL),
('TECH1', 'IT Support User', 'IT', NULL),
('HR1', 'HR User', 'HR', NULL);

-- Insert data into PARTS and COMPONENTS for BOM examples
INSERT INTO PARTS (PART_ID, PART_NAME, PART_TYPE, UNIT_COST) VALUES
('A100', 'Desktop Computer', 'ASSEMBLY', 800.00),
('A101', 'Laptop Computer', 'ASSEMBLY', 950.00),
('A102', 'Server', 'ASSEMBLY', 1500.00),
('C201', 'Motherboard', 'COMPONENT', 150.00),
('C202', 'CPU', 'COMPONENT', 200.00),
('C203', 'RAM 16GB', 'COMPONENT', 80.00),
('C204', 'SSD 512GB', 'COMPONENT', 100.00),
('C205', 'Power Supply', 'COMPONENT', 50.00),
('C206', 'Graphics Card', 'COMPONENT', 300.00),
('C207', 'Case', 'COMPONENT', 70.00),
('C208', 'Cooling Fan', 'COMPONENT', 25.00),
('C209', 'Network Card', 'COMPONENT', 30.00),
('C210', 'LCD Screen', 'COMPONENT', 150.00),
('C211', 'Keyboard', 'COMPONENT', 30.00),
('C212', 'Touchpad', 'COMPONENT', 20.00),
('C213', 'Battery', 'COMPONENT', 90.00),
('P301', 'Screw Set', 'PART', 5.00),
('P302', 'Cable Set', 'PART', 15.00),
('P303', 'Thermal Paste', 'PART', 8.00);

-- Build Desktop Computer BOM
INSERT INTO COMPONENTS (PART_ID, COMPONENT_ID, COMPONENT_QTY) VALUES
('A100', 'C201', 1),  -- Motherboard
('A100', 'C202', 1),  -- CPU
('A100', 'C203', 2),  -- RAM 16GB (2 units)
('A100', 'C204', 1),  -- SSD 512GB
('A100', 'C205', 1),  -- Power Supply
('A100', 'C206', 1),  -- Graphics Card
('A100', 'C207', 1),  -- Case
('A100', 'C208', 3),  -- Cooling Fan (3 units)
('A100', 'P301', 1),  -- Screw Set
('A100', 'P302', 1),  -- Cable Set
('A100', 'P303', 1);  -- Thermal Paste

-- Build Laptop Computer BOM
INSERT INTO COMPONENTS (PART_ID, COMPONENT_ID, COMPONENT_QTY) VALUES
('A101', 'C201', 1),  -- Motherboard
('A101', 'C202', 1),  -- CPU
('A101', 'C203', 1),  -- RAM 16GB
('A101', 'C204', 1),  -- SSD 512GB
('A101', 'C205', 1),  -- Power Supply
('A101', 'C210', 1),  -- LCD Screen
('A101', 'C211', 1),  -- Keyboard
('A101', 'C212', 1),  -- Touchpad
('A101', 'C213', 1),  -- Battery
('A101', 'C208', 1),  -- Cooling Fan
('A101', 'P301', 1),  -- Screw Set
('A101', 'P302', 1);  -- Cable Set

-- Build Server BOM
INSERT INTO COMPONENTS (PART_ID, COMPONENT_ID, COMPONENT_QTY) VALUES
('A102', 'C201', 1),  -- Motherboard
('A102', 'C202', 2),  -- CPU (2 units)
('A102', 'C203', 4),  -- RAM 16GB (4 units)
('A102', 'C204', 4),  -- SSD 512GB (4 units in RAID)
('A102', 'C205', 2),  -- Power Supply (redundant)
('A102', 'C207', 1),  -- Case
('A102', 'C208', 6),  -- Cooling Fan (6 units)
('A102', 'C209', 2),  -- Network Card (2 units)
('A102', 'P301', 2),  -- Screw Set (2 units)
('A102', 'P302', 2),  -- Cable Set (2 units)
('A102', 'P303', 2);  -- Thermal Paste (2 units)

-- Insert BUDGET data
INSERT INTO BUDGET (DEPTNO, EXPENSE_CATEGORY, FISCAL_YEAR, FISCAL_QUARTER, BUDGETED_AMOUNT) VALUES
('D01', 'Salaries', 2023, 1, 300000.00),
('D01', 'Salaries', 2023, 2, 300000.00),
('D01', 'Salaries', 2023, 3, 300000.00),
('D01', 'Salaries', 2023, 4, 300000.00),
('D01', 'Travel', 2023, 1, 50000.00),
('D01', 'Travel', 2023, 2, 55000.00),
('D01', 'Travel', 2023, 3, 60000.00),
('D01', 'Travel', 2023, 4, 45000.00),
('D02', 'Salaries', 2023, 1, 200000.00),
('D02', 'Salaries', 2023, 2, 200000.00),
('D02', 'Salaries', 2023, 3, 200000.00),
('D02', 'Salaries', 2023, 4, 200000.00),
('D02', 'Software', 2023, 1, 15000.00),
('D02', 'Software', 2023, 2, 5000.00),
('D02', 'Software', 2023, 3, 5000.00),
('D02', 'Software', 2023, 4, 15000.00),
('D03', 'Salaries', 2023, 1, 180000.00),
('D03', 'Salaries', 2023, 2, 180000.00),
('D03', 'Salaries', 2023, 3, 180000.00),
('D03', 'Salaries', 2023, 4, 180000.00),
('D03', 'Advertising', 2023, 1, 75000.00),
('D03', 'Advertising', 2023, 2, 85000.00),
('D03', 'Advertising', 2023, 3, 95000.00),
('D03', 'Advertising', 2023, 4, 120000.00),
('D04', 'Salaries', 2023, 1, 170000.00),
('D04', 'Salaries', 2023, 2, 170000.00),
('D04', 'Salaries', 2023, 3, 170000.00),
('D04', 'Salaries', 2023, 4, 170000.00),
('D04', 'Commission', 2023, 1, 50000.00),
('D04', 'Commission', 2023, 2, 55000.00),
('D04', 'Commission', 2023, 3, 60000.00),
('D04', 'Commission', 2023, 4, 75000.00),
('D05', 'Salaries', 2023, 1, 200000.00),
('D05', 'Salaries', 2023, 2, 200000.00),
('D05', 'Salaries', 2023, 3, 200000.00),
('D05', 'Salaries', 2023, 4, 200000.00),
('D05', 'Hardware', 2023, 1, 80000.00),
('D05', 'Hardware', 2023, 2, 25000.00),
('D05', 'Hardware', 2023, 3, 25000.00),
('D05', 'Hardware', 2023, 4, 70000.00);

-- Insert ACTUAL_EXPENSES data
INSERT INTO ACTUAL_EXPENSES (DEPTNO, EXPENSE_CATEGORY, FISCAL_YEAR, FISCAL_QUARTER, ACTUAL_AMOUNT) VALUES
('D01', 'Salaries', 2023, 1, 298500.00),
('D01', 'Salaries', 2023, 2, 301200.00),
('D01', 'Salaries', 2023, 3, 304800.00),
('D01', 'Travel', 2023, 1, 42500.00),
('D01', 'Travel', 2023, 2, 61300.00),
('D01', 'Travel', 2023, 3, 57200.00),
('D02', 'Salaries', 2023, 1, 199200.00),
('D02', 'Salaries', 2023, 2, 201500.00),
('D02', 'Salaries', 2023, 3, 200800.00),
('D02', 'Software', 2023, 1, 16200.00),
('D02', 'Software', 2023, 2, 4800.00),
('D02', 'Software', 2023, 3, 5500.00),
('D03', 'Salaries', 2023, 1, 177500.00),
('D03', 'Salaries', 2023, 2, 179800.00),
('D03', 'Salaries', 2023, 3, 182300.00),
('D03', 'Advertising', 2023, 1, 72300.00),
('D03', 'Advertising', 2023, 2, 89500.00),
('D03', 'Advertising', 2023, 3, 97800.00),
('D04', 'Salaries', 2023, 1, 169500.00),
('D04', 'Salaries', 2023, 2, 168200.00),
('D04', 'Salaries', 2023, 3, 171500.00),
('D04', 'Commission', 2023, 1, 47800.00),
('D04', 'Commission', 2023, 2, 58200.00),
('D04', 'Commission', 2023, 3, 64500.00),
('D05', 'Salaries', 2023, 1, 197500.00),
('D05', 'Salaries', 2023, 2, 201200.00),
('D05', 'Salaries', 2023, 3, 204500.00),
('D05', 'Hardware', 2023, 1, 87300.00),
('D05', 'Hardware', 2023, 2, 21500.00),
('D05', 'Hardware', 2023, 3, 26800.00),
-- New expense not in budget
('D03', 'Event Sponsorship', 2023, 2, 25000.00),
('D04', 'Training', 2023, 3, 15000.00);

-- Populate DAILY_SALES for time series analysis
-- Clear existing data if needed
DELETE FROM DAILY_SALES;
 
-- Create a temporary sequence table
CREATE TABLE DEMOSCHEMA.TEMP_SEQUENCE (
  SEQ_NUM INTEGER NOT NULL PRIMARY KEY
);

-- Insert 180 sequence numbers
INSERT INTO DEMOSCHEMA.TEMP_SEQUENCE (SEQ_NUM) VALUES
(1), (2), (3), (4), (5), (6), (7), (8), (9), (10),
(11), (12), (13), (14), (15), (16), (17), (18), (19), (20),
(21), (22), (23), (24), (25), (26), (27), (28), (29), (30),
(31), (32), (33), (34), (35), (36), (37), (38), (39), (40),
(41), (42), (43), (44), (45), (46), (47), (48), (49), (50),
(51), (52), (53), (54), (55), (56), (57), (58), (59), (60),
(61), (62), (63), (64), (65), (66), (67), (68), (69), (70),
(71), (72), (73), (74), (75), (76), (77), (78), (79), (80),
(81), (82), (83), (84), (85), (86), (87), (88), (89), (90),
(91), (92), (93), (94), (95), (96), (97), (98), (99), (100),
(101), (102), (103), (104), (105), (106), (107), (108), (109), (110),
(111), (112), (113), (114), (115), (116), (117), (118), (119), (120),
(121), (122), (123), (124), (125), (126), (127), (128), (129), (130),
(131), (132), (133), (134), (135), (136), (137), (138), (139), (140),
(141), (142), (143), (144), (145), (146), (147), (148), (149), (150),
(151), (152), (153), (154), (155), (156), (157), (158), (159), (160),
(161), (162), (163), (164), (165), (166), (167), (168), (169), (170),
(171), (172), (173), (174), (175), (176), (177), (178), (179), (180);

-- Create a temporary dates table with all the dates we need
CREATE TABLE DEMOSCHEMA.TEMP_DATES (
  SALE_DATE DATE NOT NULL PRIMARY KEY,
  DAY_TYPE VARCHAR(10),
  SEQ_NUMBER INTEGER
);

-- Insert all dates and categorize them
INSERT INTO DEMOSCHEMA.TEMP_DATES (SALE_DATE, DAY_TYPE, SEQ_NUMBER)
SELECT 
  DATE('2025-03-29') - (180 - SEQ_NUM) DAYS AS SALE_DATE,
  CASE
    WHEN DAYOFWEEK(DATE('2025-03-29') - (180 - SEQ_NUM) DAYS) IN (1, 7) THEN 'WEEKEND'
    WHEN DAY(DATE('2025-03-29') - (180 - SEQ_NUM) DAYS) BETWEEN 25 AND 31 THEN 'MONTH_END'
    ELSE 'WEEKDAY'
  END AS DAY_TYPE,
  SEQ_NUM
FROM DEMOSCHEMA.TEMP_SEQUENCE
WHERE SEQ_NUM <= 180;

-- Now insert weekend data
INSERT INTO DAILY_SALES (ORDERDATE, TOTAL_AMOUNT, ORDER_COUNT, CUSTOMER_COUNT)
SELECT 
  SALE_DATE,
  5000 + (SEQ_NUMBER * 75) AS TOTAL_AMOUNT,
  15 + MOD(SEQ_NUMBER, 10) AS ORDER_COUNT,
  10 + MOD(SEQ_NUMBER, 8) AS CUSTOMER_COUNT
FROM DEMOSCHEMA.TEMP_DATES
WHERE DAY_TYPE = 'WEEKEND';

-- Insert month-end data
INSERT INTO DAILY_SALES (ORDERDATE, TOTAL_AMOUNT, ORDER_COUNT, CUSTOMER_COUNT)
SELECT 
  SALE_DATE,
  12000 + (SEQ_NUMBER * 120) AS TOTAL_AMOUNT,
  35 + MOD(SEQ_NUMBER, 15) AS ORDER_COUNT,
  25 + MOD(SEQ_NUMBER, 10) AS CUSTOMER_COUNT
FROM DEMOSCHEMA.TEMP_DATES
WHERE DAY_TYPE = 'MONTH_END';

-- Insert regular weekday data
INSERT INTO DAILY_SALES (ORDERDATE, TOTAL_AMOUNT, ORDER_COUNT, CUSTOMER_COUNT)
SELECT 
  SALE_DATE,
  8000 + (SEQ_NUMBER * 100) AS TOTAL_AMOUNT,
  25 + MOD(SEQ_NUMBER, 12) AS ORDER_COUNT,
  18 + MOD(SEQ_NUMBER, 9) AS CUSTOMER_COUNT
FROM DEMOSCHEMA.TEMP_DATES
WHERE DAY_TYPE = 'WEEKDAY';

-- Add variation to the data to make it more realistic
UPDATE DAILY_SALES 
SET 
  TOTAL_AMOUNT = CAST(TOTAL_AMOUNT * (1 + (MOD(DAYS(ORDERDATE), 7) / 100.0)) AS DECIMAL(12,2)),
  ORDER_COUNT = CAST(ORDER_COUNT * (1 + (MOD(DAYS(ORDERDATE), 5) / 100.0)) AS INTEGER),
  CUSTOMER_COUNT = CAST(CUSTOMER_COUNT * (1 + (MOD(DAYS(ORDERDATE), 3) / 100.0)) AS INTEGER);

-- Clean up temporary tables
DROP TABLE DEMOSCHEMA.TEMP_SEQUENCE;
DROP TABLE DEMOSCHEMA.TEMP_DATES;

-- Verify the data
SELECT 'DAILY_SALES populated successfully with ' || 
       CAST(COUNT(*) AS VARCHAR(10)) || ' days of data' AS RESULT
FROM DAILY_SALES;

-- Display data distribution by day of week
SELECT 
  DAYOFWEEK(ORDERDATE) AS DAY_OF_WEEK,
  COUNT(*) AS DAY_COUNT,
  AVG(TOTAL_AMOUNT) AS AVG_AMOUNT,
  MIN(TOTAL_AMOUNT) AS MIN_AMOUNT,
  MAX(TOTAL_AMOUNT) AS MAX_AMOUNT
FROM DAILY_SALES
GROUP BY DAYOFWEEK(ORDERDATE)
ORDER BY DAYOFWEEK(ORDERDATE);

-- Display data distribution by day of month
SELECT 
  CASE 
    WHEN DAY(ORDERDATE) BETWEEN 1 AND 10 THEN 'Days 1-10'
    WHEN DAY(ORDERDATE) BETWEEN 11 AND 20 THEN 'Days 11-20'
    ELSE 'Days 21-31'
  END AS MONTH_PERIOD,
  COUNT(*) AS DAY_COUNT,
  AVG(TOTAL_AMOUNT) AS AVG_AMOUNT,
  MIN(TOTAL_AMOUNT) AS MIN_AMOUNT,
  MAX(TOTAL_AMOUNT) AS MAX_AMOUNT
FROM DAILY_SALES
GROUP BY 
  CASE 
    WHEN DAY(ORDERDATE) BETWEEN 1 AND 10 THEN 'Days 1-10'
    WHEN DAY(ORDERDATE) BETWEEN 11 AND 20 THEN 'Days 11-20'
    ELSE 'Days 21-31'
  END
ORDER BY 
  CASE 
    WHEN MONTH_PERIOD = 'Days 1-10' THEN 1
    WHEN MONTH_PERIOD = 'Days 11-20' THEN 2
    ELSE 3
  END;
-- Populate CUSTOMER_STATUS table
INSERT INTO CUSTOMER_STATUS (CUSTNO, CUSTNAME, LAST_ORDER_DATE, ANNUAL_SPEND, STATUS, LAST_UPDATED)
SELECT 
  c.CUSTNO,
  c.CUSTNAME,
  (SELECT MAX(ORDERDATE) FROM ORDERS WHERE CUSTNO = c.CUSTNO) AS LAST_ORDER_DATE,
  (SELECT SUM(TOTAL_AMOUNT) FROM ORDERS WHERE CUSTNO = c.CUSTNO AND ORDERDATE >= CURRENT DATE - 1 YEAR) AS ANNUAL_SPEND,
  c.STATUS,
  CURRENT TIMESTAMP AS LAST_UPDATED
FROM CUSTOMER c;

-- Create sample JSON for the JSON_ORDERS table
INSERT INTO JSON_ORDERS (ORDERNO, ORDER_JSON)
VALUES (12345, JSON_OBJECT(
  'orderNo' VALUE 12345,
  'orderDate' VALUE '2023-03-15',
  'customer' VALUE JSON_OBJECT(
    'id' VALUE 'C000000003',
    'name' VALUE 'Tech Solutions',
    'contact' VALUE JSON_OBJECT(
      'email' VALUE 'sales@techsolutions.com',
      'phone' VALUE '415-555-9012'
    )
  ),
  'items' VALUE JSON_ARRAY(
    JSON_OBJECT(
      'productId' VALUE 'P0001',
      'productName' VALUE 'Laptop Pro',
      'quantity' VALUE 5,
      'unitPrice' VALUE 1299.99,
      'lineTotal' VALUE 6499.95
    ),
    JSON_OBJECT(
      'productId' VALUE 'P0003',
      'productName' VALUE 'Wireless Headphones',
      'quantity' VALUE 10,
      'unitPrice' VALUE 199.99,
      'lineTotal' VALUE 1999.90
    ),
    JSON_OBJECT(
      'productId' VALUE 'P0009',
      'productName' VALUE 'Wireless Keyboard',
      'quantity' VALUE 5,
      'unitPrice' VALUE 89.99,
      'lineTotal' VALUE 449.95
    )
  ),
  'total' VALUE 8949.80,
  'status' VALUE 'COMPLETED',
  'shippingAddress' VALUE JSON_OBJECT(
    'street' VALUE '789 Market St',
    'city' VALUE 'San Francisco',
    'state' VALUE 'CA',
    'zipCode' VALUE '94103',
    'country' VALUE 'USA'
  ),
  'shippingMethod' VALUE 'Express',
  'paymentMethod' VALUE 'Credit Card',
  'notes' VALUE 'Corporate office equipment update'
));

INSERT INTO JSON_ORDERS (ORDERNO, ORDER_JSON)
VALUES (12346, JSON_OBJECT(
  'orderNo' VALUE 12346,
  'orderDate' VALUE '2023-03-18',
  'customer' VALUE JSON_OBJECT(
    'id' VALUE 'C000000005',
    'name' VALUE 'Innovative Labs',
    'contact' VALUE JSON_OBJECT(
      'email' VALUE 'contact@innovativelabs.com',
      'phone' VALUE '206-555-7890'
    )
  ),
  'items' VALUE JSON_ARRAY(
    JSON_OBJECT(
      'productId' VALUE 'P0002',
      'productName' VALUE 'Smartphone X',
      'quantity' VALUE 3,
      'unitPrice' VALUE 899.99,
      'lineTotal' VALUE 2699.97
    ),
    JSON_OBJECT(
      'productId' VALUE 'P0004',
      'productName' VALUE 'Smart Watch',
      'quantity' VALUE 3,
      'unitPrice' VALUE 349.99,
      'lineTotal' VALUE 1049.97
    )
  ),
  'total' VALUE 3749.94,
  'status' VALUE 'SHIPPED',
  'shippingAddress' VALUE JSON_OBJECT(
    'street' VALUE '654 5th Ave',
    'city' VALUE 'Seattle',
    'state' VALUE 'WA',
    'zipCode' VALUE '98101',
    'country' VALUE 'USA'
  ),
  'shippingMethod' VALUE 'Standard',
  'paymentMethod' VALUE 'Purchase Order',
  'notes' VALUE 'Executive team devices'
));

-- Create sample JSON for PRODUCT_CATALOG
INSERT INTO PRODUCT_CATALOG (PRODUCT_ID, PRODUCT_NAME, PRODUCT_DETAILS)
VALUES ('LP001', 'Laptop Pro X1', JSON_OBJECT(
  'manufacturer' VALUE 'TechCorp',
  'category' VALUE 'Computers',
  'description' VALUE 'Professional grade laptop with high performance specifications',
  'price' VALUE 1499.99,
  'specifications' VALUE JSON_OBJECT(
    'processor' VALUE 'Intel Core i7 12th Gen',
    'ram' VALUE '32GB DDR5',
    'storage' VALUE '1TB SSD',
    'display' VALUE '15.6" 4K OLED',
    'graphics' VALUE 'NVIDIA RTX 3070',
    'weight' VALUE 1.8,
    'dimensions' VALUE JSON_OBJECT(
      'height' VALUE 0.7,
      'width' VALUE 13.3,
      'depth' VALUE 9.2
    ),
    'battery' VALUE '95Wh',
    'ports' VALUE JSON_ARRAY('USB-C x2', 'USB-A x3', 'HDMI', 'Ethernet', 'Audio Jack')
  ),
  'features' VALUE JSON_ARRAY(
    'Backlit Keyboard',
    'Fingerprint Reader',
    'HD Webcam',
    'Thunderbolt 4 Support',
    'Wi-Fi 6E'
  ),
  'warranty' VALUE '3 Years Premium Support',
  'inStock' VALUE true,
  'rating' VALUE 4.8,
  'reviews' VALUE 256
));

INSERT INTO PRODUCT_CATALOG (PRODUCT_ID, PRODUCT_NAME, PRODUCT_DETAILS)
VALUES ('SP001', 'Smartphone Pro', JSON_OBJECT(
  'manufacturer' VALUE 'Apple',
  'category' VALUE 'Phones',
  'description' VALUE 'Latest flagship smartphone with advanced camera system',
  'price' VALUE 999.99,
  'specifications' VALUE JSON_OBJECT(
    'processor' VALUE 'A15 Bionic',
    'ram' VALUE '8GB',
    'storage' VALUE '256GB',
    'display' VALUE '6.1" Super Retina XDR',
    'camera' VALUE 'Triple 12MP Ultra Wide',
    'weight' VALUE 0.174,
    'dimensions' VALUE JSON_OBJECT(
      'height' VALUE 146.7,
      'width' VALUE 71.5,
      'depth' VALUE 7.65
    ),
    'battery' VALUE '3095mAh',
    'biometrics' VALUE 'Face ID'
  ),
  'features' VALUE JSON_ARRAY(
    '5G Connectivity',
    'Water Resistance IP68',
    'Wireless Charging',
    'Night Mode Photography',
    'Spatial Audio'
  ),
  'warranty' VALUE '1 Year Limited Warranty',
  'inStock' VALUE true,
  'rating' VALUE 4.9,
  'reviews' VALUE 1024
));

INSERT INTO PRODUCT_CATALOG (PRODUCT_ID, PRODUCT_NAME, PRODUCT_DETAILS)
VALUES ('TB001', 'Tablet Pro 12', JSON_OBJECT(
  'manufacturer' VALUE 'Samsung',
  'category' VALUE 'Tablets',
  'description' VALUE 'Premium tablet with S Pen support for productivity',
  'price' VALUE 799.99,
  'specifications' VALUE JSON_OBJECT(
    'processor' VALUE 'Snapdragon 8 Gen 1',
    'ram' VALUE '8GB',
    'storage' VALUE '256GB',
    'display' VALUE '12.4" AMOLED 120Hz',
    'camera' VALUE '12MP Wide + 6MP Ultrawide',
    'weight' VALUE 0.567,
    'dimensions' VALUE JSON_OBJECT(
      'height' VALUE 285,
      'width' VALUE 185,
      'depth' VALUE 5.7
    ),
    'battery' VALUE '10090mAh'
  ),
  'features' VALUE JSON_ARRAY(
    'S Pen Included',
    'DeX Mode Support',
    'AKG Quad Speakers',
    'Fast Charging',
    'Expandable Storage'
  ),
  'warranty' VALUE '2 Year Warranty',
  'inStock' VALUE true,
  'rating' VALUE 4.7,
  'reviews' VALUE 512
));

-- Create some views for common queries
CREATE OR REPLACE VIEW EMPLOYEE_SUMMARY AS
SELECT 
  d.DEPTNO,
  d.DEPTNAME,
  e.EMPNO,
  e.LASTNAME,
  e.FIRSTNAME,
  e.JOB_TITLE,
  e.SALARY,
  m.LASTNAME AS MANAGER,
  e.HIREDATE,
  e.EMAIL
FROM EMPLOYEE e
JOIN DEPARTMENT d ON e.DEPTNO = d.DEPTNO
LEFT JOIN EMPLOYEE m ON e.MANAGERNO = m.EMPNO;

CREATE OR REPLACE VIEW CUSTOMER_ORDERS_SUMMARY AS
SELECT 
  c.CUSTNO,
  c.CUSTNAME,
  r.REGION_NAME,
  e.LASTNAME AS SALES_REP,
  COUNT(o.ORDERNO) AS ORDER_COUNT,
  SUM(o.TOTAL_AMOUNT) AS TOTAL_AMOUNT,
  AVG(o.TOTAL_AMOUNT) AS AVG_ORDER_AMOUNT,
  MAX(o.ORDERDATE) AS LAST_ORDER_DATE,
  MIN(o.ORDERDATE) AS FIRST_ORDER_DATE
FROM CUSTOMER c
LEFT JOIN ORDERS o ON c.CUSTNO = o.CUSTNO
LEFT JOIN REGION r ON c.REGION_ID = r.REGION_ID
LEFT JOIN EMPLOYEE e ON c.SALES_REP_ID = e.EMPNO
GROUP BY c.CUSTNO, c.CUSTNAME, r.REGION_NAME, e.LASTNAME;

CREATE OR REPLACE VIEW PRODUCT_SALES_SUMMARY AS
SELECT 
  p.PRODNO,
  p.PRODNAME,
  p.CATEGORY,
  SUM(ol.QUANTITY) AS TOTAL_QUANTITY,
  SUM(ol.QUANTITY * ol.UNIT_PRICE) AS TOTAL_REVENUE,
  COUNT(DISTINCT o.ORDERNO) AS ORDER_COUNT,
  COUNT(DISTINCT o.CUSTNO) AS CUSTOMER_COUNT,
  p.UNITPRICE - p.COST_PRICE AS UNIT_PROFIT,
  (p.UNITPRICE - p.COST_PRICE) / p.UNITPRICE * 100 AS PROFIT_MARGIN
FROM PRODUCT p
LEFT JOIN ORDERLINE ol ON p.PRODNO = ol.PRODNO
LEFT JOIN ORDERS o ON ol.ORDERNO = o.ORDERNO
GROUP BY p.PRODNO, p.PRODNAME, p.CATEGORY, p.UNITPRICE, p.COST_PRICE;

-- Create indexes for performance
CREATE INDEX ix_employee_deptno ON EMPLOYEE (DEPTNO);
CREATE INDEX ix_employee_mgr ON EMPLOYEE (MANAGERNO);
CREATE INDEX ix_orders_date ON ORDERS (ORDERDATE);
CREATE INDEX ix_orders_cust ON ORDERS (CUSTNO);
CREATE INDEX ix_orderline_prod ON ORDERLINE (PRODNO);
CREATE INDEX ix_customer_region ON CUSTOMER (REGION_ID);
CREATE INDEX ix_customer_status ON CUSTOMER (STATUS);
CREATE INDEX ix_product_category ON PRODUCT (CATEGORY);
CREATE INDEX ix_customer_sales_rep ON CUSTOMER (SALES_REP_ID);

-- Procedure to analyze order trends
CREATE PROCEDURE ANALYZE_ORDER_TRENDS 
(
  P_START_DATE DATE,
  P_END_DATE DATE,
  P_REGION_ID CHAR(2)
)
RESULT SETS 1
BEGIN
  -- Declare cursor for monthly trends
  DECLARE CURSOR1 CURSOR FOR
    SELECT 
      YEAR(O.ORDERDATE) AS ORDER_YEAR,
      MONTH(O.ORDERDATE) AS ORDER_MONTH,
      COUNT(O.ORDERNO) AS ORDER_COUNT,
      SUM(O.TOTAL_AMOUNT) AS TOTAL_SALES,
      COUNT(DISTINCT O.CUSTNO) AS CUSTOMER_COUNT
    FROM ORDERS O
    JOIN CUSTOMER C ON O.CUSTNO = C.CUSTNO
    WHERE (P_START_DATE IS NULL OR O.ORDERDATE >= P_START_DATE)
    AND (P_END_DATE IS NULL OR O.ORDERDATE <= P_END_DATE)
    AND (P_REGION_ID IS NULL OR C.REGION_ID = P_REGION_ID)
    GROUP BY YEAR(O.ORDERDATE), MONTH(O.ORDERDATE)
    ORDER BY YEAR(O.ORDERDATE), MONTH(O.ORDERDATE);
  
  -- Open cursor to return results
  OPEN CURSOR1;
END;

-- Simplified Product Recommendations Function for DB2 for i
CREATE OR REPLACE FUNCTION RECOMMENDED_PRODUCTS (
  P_CUSTNO CHAR(10),
  P_LIMIT INTEGER
)
RETURNS TABLE (
  PRODNO CHAR(10),
  PRODNAME VARCHAR(100),
  CATEGORY VARCHAR(50),
  RECOMMENDATION_TYPE VARCHAR(20),
  SCORE DECIMAL(5,2)
)
LANGUAGE SQL
DETERMINISTIC
NO EXTERNAL ACTION
READS SQL DATA
RETURN
  -- Use a simpler approach with just two recommendation types
  -- 1. Category-based recommendations from customer's purchase history
  SELECT 
    P.PRODNO,
    P.PRODNAME,
    P.CATEGORY,
    'CATEGORY_BASED' AS RECOMMENDATION_TYPE,
    5.0 AS SCORE
  FROM PRODUCT P
  WHERE P.CATEGORY IN (
    -- Find categories this customer has purchased from
    SELECT DISTINCT PR.CATEGORY 
    FROM ORDERS O
    JOIN ORDERLINE OL ON O.ORDERNO = OL.ORDERNO
    JOIN PRODUCT PR ON OL.PRODNO = PR.PRODNO
    WHERE O.CUSTNO = P_CUSTNO
  )
  -- Exclude products the customer already purchased
  AND P.PRODNO NOT IN (
    SELECT OL.PRODNO
    FROM ORDERS O
    JOIN ORDERLINE OL ON O.ORDERNO = OL.ORDERNO
    WHERE O.CUSTNO = P_CUSTNO
  )
  AND P.ACTIVE_FLAG = 'Y'
  
  UNION ALL
  
  -- 2. Popular products recommendation (most-ordered products)
  SELECT 
    P.PRODNO,
    P.PRODNAME,
    P.CATEGORY,
    'POPULAR_PRODUCT' AS RECOMMENDATION_TYPE,
    4.0 AS SCORE
  FROM PRODUCT P
  JOIN (
    SELECT 
      OL.PRODNO,
      SUM(OL.QUANTITY) AS TOTAL_ORDERED
    FROM ORDERLINE OL
    JOIN ORDERS O ON OL.ORDERNO = O.ORDERNO
    GROUP BY OL.PRODNO
    ORDER BY TOTAL_ORDERED DESC
    FETCH FIRST 20 ROWS ONLY
  ) AS POPULAR ON P.PRODNO = POPULAR.PRODNO
  -- Exclude products the customer already purchased
  WHERE P.PRODNO NOT IN (
    SELECT OL.PRODNO
    FROM ORDERS O
    JOIN ORDERLINE OL ON O.ORDERNO = OL.ORDERNO
    WHERE O.CUSTNO = P_CUSTNO
  )
  AND P.ACTIVE_FLAG = 'Y'
  
  -- Sort by score and limit results
  ORDER BY SCORE DESC, PRODNO
  FETCH FIRST P_LIMIT ROWS ONLY;

-- Example usage:
SELECT * FROM TABLE(DEMOSCHEMA.RECOMMENDED_PRODUCTS(P_CUSTNO => 'C000000001', P_LIMIT => 5));

-- Create a user-defined function for customer lifetime value calculation
CREATE  FUNCTION customer_lifetime_value(
  p_custno CHAR(10),
  p_discount_rate DECIMAL(5,2) DEFAULT 10.00  -- Annual discount rate as a percentage
)
RETURNS DECIMAL(12,2)
LANGUAGE SQL
DETERMINISTIC

RETURN
  WITH yearly_revenue AS (
    SELECT 
      YEAR(ORDERDATE) AS YEAR,
      SUM(TOTAL_AMOUNT) AS ANNUAL_REVENUE
    FROM ORDERS
    WHERE CUSTNO = p_custno
    GROUP BY YEAR(ORDERDATE)
  ),
  lifetime_metrics AS (
    SELECT 
      COUNT(*) AS YEARS_ACTIVE,
      AVG(ANNUAL_REVENUE) AS AVG_ANNUAL_REVENUE,
      (SELECT COUNT(*) FROM ORDERS WHERE CUSTNO = p_custno) AS TOTAL_ORDERS,
      (SELECT MAX(ORDERDATE) FROM ORDERS WHERE CUSTNO = p_custno) AS LAST_ORDER_DATE
    FROM yearly_revenue
  )
  SELECT 
    CASE
      WHEN DAYS(CURRENT DATE) - DAYS(LAST_ORDER_DATE) > 365 THEN 
        -- Customer likely churned, return historical value
        AVG_ANNUAL_REVENUE * YEARS_ACTIVE
      ELSE 
        -- Active customer, project future value
        AVG_ANNUAL_REVENUE * (1 + (100 - p_discount_rate) / 100)
    END
  FROM lifetime_metrics;
 

-- Final message
SELECT 'DB2 for i database schema successfully created and populated with sample data' AS MESSAGE FROM SYSIBM.SYSDUMMY1;
 